
function H = setuppyramid(responses, RFangles, layerI, layerJ, filtI, filtJ, ...
                          minI, minJ, maxI, maxJ);

% FUNCTION H = SETUPPYRAMID(responses, RFangles, layerI, layerJ, 
%                           filtI, filtJ, minI, minJ, maxI, maxJ);
%
% Sets up the pyramid structure used for attention and initializes the
% various parameters and variables.
% Returns in H a matlab structure storing the various pieces of information.
% Inputs are :
%  responses is a 3-d array of the edge responses returned by function EDGES.
%  RFangles is a list of receptive field angles.  This is usually the same
%   as the angles input passed to the function EDGES.
%  layerI is a list of number of rows for each layer in the pyramid.
%  layerJ is a list of number of columns for each layer in the pyramid.
%  filtI is a list of number of rows for the averaging filters used to
%   uppropagate the responses values up the pyramid.
%  filtJ is a list of number of columns for the averaging filters used to
%   uppropagate the responses values up the pyramid.
%  minI, minJ are lists of the minimum number of rows/columns for each 
%   receptive field in the different layers.
%  maxI, maxJ are lists of the maximum number of rows/columns for each 
%   receptive field in the different layers.

% Standardize the RFangles.
RFangles = mod(RFangles,180);

% Number of edge features.
F        = length(RFangles);

% Number of layers.
L        = length(layerI);

% Construct the averaging filters used for uppropagate.
filt = cell(1,L-1);
for l=1:L-1,
 filt{l} = ones(filtI(l),filtJ(l))/(filtI(l)*filtJ(l));
end;

% Computes the separation between each application of the averaging filter.
skipI = (layerI(1:end-1)-filtI)./(layerI(2:end)-1);
skipJ = (layerJ(1:end-1)-filtJ)./(layerJ(2:end)-1);

D              = cell(L,F);	% Stores the responses.
S              = cell(L,F);	% Stores the sheered responses.
sheer          = cell(L,F);	% The amount of sheering done to each row/col.
sheerI         = zeros(L,F);	% The resulting number of rows and columns
sheerJ         = zeros(L,F);	%  of the sheered responses.
sheerangles    = zeros(1,F);	% The angle of sheering.
sheerflipud    = zeros(1,F);	% Whether we should flip the feature map 
				%  vertically.
sheertranspose = zeros(1,F);	% Whether we should transpose the feature map.
for f=1:F,
 sheerflipud(f) = (RFangles(f)>90);
 if sheerflipud(f), 
  sheerangles(f) = 180-RFangles(f);
 else
  sheerangles(f) = RFangles(f);
 end;
 sheertranspose(f) = (sheerangles(f)>45);
 if sheertranspose(f), 
  sheerangles(f) = 90-sheerangles(f);
 end;

 for l=1:L,
  sheer{l,f} = round((0:layerI(l)-1)*tan(sheerangles(f)/180*pi));
  sheerI(l,f) = layerI(l);
  sheerJ(l,f) = layerJ(l)+sheer{l,f}(end);
  D{l,f} = zeros(layerI(l), layerJ(l));
  S{l,f} = zeros(sheerI(l)+1, sheerJ(l)+1);
 end;
 D{1,f} = abs(responses(:,:,f));
end;

% beam is a structure storing the current attention beam.
beam = cell(L,F);
for f=1:F,
 for l=1:L,
  beam{l,f} = struct('v',-Inf,'t',1,'b',1,'l',1,'r',1);
 end;
end;

H = struct( ...
 'RFangles',       {RFangles}, ...
 'F',              {F}, ...
 'L',              {L}, ...
 'layerI',         {layerI}, ...
 'layerJ',         {layerJ}, ...
 'filt',           {filt} , ...
 'filtI',          {filtI}, ...
 'filtJ',          {filtJ}, ...
 'skipI',          {skipI}, ...
 'skipJ',          {skipJ}, ...
 'D',              {D}, ...
 'S',              {S}, ...
 'sheer',          {sheer}, ...
 'sheerI',         {sheerI}, ...
 'sheerJ',         {sheerJ}, ...
 'sheerangles',    {sheerangles}, ...
 'sheerflipud',    {sheerflipud}, ...
 'sheertranspose', {sheertranspose}, ...
 'fovea',          {{}}, ...
 'beam',           {beam}, ...
 'minI',           {max(1,minI)}, ...
 'minJ',           {max(1,minJ)}, ...
 'maxI',           {min(layerI,maxI)}, ...
 'maxJ',           {min(layerJ,maxJ)});

% Uppropagate the responses.
H = uppropagate(H);
