function response = edges(image, angles);

% FUNCTION response = EDGES(image, angles);
% 
% Computes the edges present in the image.  The angles of the edges are given
% in degrees clockwise from the positive y-axis.  The responses are returned
% as a 3-d array with the i'th layer being the angles(i) edge responses of 
% the image.
%
% Uses BuildSpyr, spyrBand and steer of the steerable pyramid package of
% Simoncelli.  Filters used are sp1Filters, sp3Filters and sp5Filters,
% depending on the distribution of angles specified.

% number of rows and columns of the image.
[r c] = size(image);

% changes the image to an intensity image.
if size(image,3)==3,
 image = rgb2gray(im2double(image));
end;

% Puts angles into a standardized form (in the steerable pyramid package,
% angles are counter-clockwise about y-axis, so have to negate).
angles = mod(-angles, 360);

if size(angles)==1,
 % If only one angle specified, uses the highest frequency resolution filter.
 filters  = 'sp5Filters';
 numbases = 6;
else
 % If more than one angle, uses highest frequency resolution filter that
 % covers frequency space.
 a= sort(angles);
 mindiff = min(a(2:end)-a(1:end-1));
 if mindiff==180,
  filters  = 'sp1Filters';
  numbases = 2;
 elseif mindiff>=90,
  filters  = 'sp3Filters';
  numbases = 4;
 else
  filters  = 'sp5Filters';
  numbases = 6;
 end;
end;

% Builds the pyramid and gets basis responses (convolution of basis filters
% with images).
[p,i,s,h] = buildSpyr(image,1,filters);
baseres = zeros([size(image) numbases]);
for j=1:numbases,
 baseres(:,:,j) = spyrBand(p,i,1,j);
end;
baseres = reshape(baseres,[prod(size(image)) numbases]);

% Steers the basis responses to get the edge responses at the various angles.
response = zeros([prod(size(image)) length(angles)]);
for j=1:length(angles),
 response(:,j) = steer(baseres, angles(j)/180*pi);
end;
response = reshape(response,[size(image) length(angles)]);
