# file spatial/pp.q copyright (C) 1994-9 W. N. Venables and B. D. Ripley
#

setClass("pps", representation(x="numeric", y="numeric", area="named"),
         validity = function(object){
  if(length(object@y) !=  length(object@x))
    return("mismatch in lengths of x and y")
  else if(length(object@area) != 4)
    return("corrupt area slot")
  else return(TRUE)
})

ppinit <- function(file)
{
  tfile <- file
  if(exists(".sp.lib.name", f=0)) {
    t1file <- paste(get(".sp.lib.name", f=0), file, sep="/")
    if(file.exists(t1file)) tfile <- t1file
  }
  if(!file.exists(tfile)) stop(paste("File", file, "not found"))
  tf <- open(tfile)
  h <- scan(tf, list(xl = 0, xu = 0, yl = 0, yu = 0, fac = 0),
            n = 5, skip = 2)
  pp <- scan(tf, list(x = 0, y = 0))
  close(tf)
  pp <- new("pps", x=pp$x/h$fac, y=pp$y/h$fac,
      area=c(xl=h$xl/h$fac, xu=h$xu/h$fac, yl=h$yl/h$fac, yu=h$yu/h$fac))
  if(!is.loaded(symbol.C("VR_ppset")))
    stop("Compiled code has not been dynamically loaded")
  ppregion(pp)
  invisible(pp)
}

Kfn <- function(pp, fs, k = 100)
{
  zz <- (c(range(pp@x), range(pp@y)) - ppgetregion())*c(1,-1,1,-1)
  if(any(zz < 0)) stop("some points outside region")
  z <- .C("VR_sp_pp2",
	  pp@x, pp@y, length(pp@x), k1 = k, h = single(k),
	  dmin = single(1), lm = single(1), fs)
  list(y = z$h[1:z$k1], x = (seq(1:z$k1) * fs)/k, k = k,
       dmin = z$dmin, lm = max(z$dmin, z$lm),
       call=match.call())
}

Kenvl <- function(fs, nsim, ...)
{
  dot.expression <- as.expression(substitute(list(...)))[-1]
  h <- Kfn(pp = eval(dot.expression), fs)
  hx <- h$x
  hu <- h$y
  hl <- h$y
  ha <- h$y^2
  for(i in 2:nsim) {
    h <- Kfn(pp = eval(dot.expression), fs)$y
    hu <- pmax(hu, h)
    hl <- pmin(hl, h)
    ha <- ha + h^2
  }
  list(x = hx, lower = hl, upper = hu, aver = sqrt(ha/nsim),
       call=match.call())
}

Kaver <- function(fs, nsim, ...)
{
  dot.expression <- as.expression(substitute(list(...)))[-1]
  h <- Kfn(pp = eval(dot.expression), fs)
  hx <- h$x
  ha <- h$y^2
  for(i in 2:nsim) {
    h <- Kfn(pp = eval(dot.expression), fs)$y
    ha <- ha + h^2
  }
  list(x = hx, y = sqrt(ha/nsim), call=match.call())
}

ppregion <- function(xl = 0, xu = 1, yl = 0, yu = 1)
{
    if(is.null(xl)) stop("invalid input")
    if(is.numeric(xl))
        if(length(xl) != 1 || length(xu) != 1 ||
           length(yl) != 1 || length(yu) != 1)
            stop("invalid input")
    if(is.list(xl)) {
        if(is.null(xl$area) &&
           any(is.na(match( c("xl", "xu", "yl", "yu"), names(xl)))))
            stop("invalid input")
    }
    if(inherits(xl, "pps"))  .C("VR_ppset", xl@area)
    else  .C("VR_ppset", c(xl, xu, yl, yu))
    invisible()
}

ppgetregion <- function()
{
    xx <- .C("VR_ppget", z=single(4))$z
    names(xx) <- c("xl", "xu", "yl", "yu")
    xx
}

Psim <- function(n)
{
  z <- .C("VR_pdata", n, x = single(n), y = single(n))
  invisible(list(x = z@x, y = z@y, call=match.call()))
}

Strauss <- function(n, c = 0, r)
{
  init <-  0
  if(!exists(".ppx", frame=1)) {
    init <-  1
    z <- .C("VR_pdata", n, x = single(n), y = single(n))
    assign(".ppx", z@x, frame=1)
    assign(".ppy", z@y, frame=1)
  }
  z <- .C("VR_simpat", n, x = .ppx, y = .ppy, c, r, init)
  assign(".ppx", z@x, frame=1)
  assign(".ppy", z@y, frame=1)
  invisible(list(x = z@x, y = z@y, call=match.call()))
}

SSI <- function(n, r)
{
  z <- .C("VR_simmat", n, x = single(n), y = single(n), r)
  invisible(list(x = z@x, y = z@y, call=match.call()))
}


pplik <- function(pp, R, ng=50, trace=F)
{
    pplikfn <- function(cc, R, n, x, y, ng, target, trace=F)
    {
        z <- .C("VR_plike", x, y, n, cc, R, ng, target, res=single(1))
        if(trace) print(c(cc, z$res))
        z$res
    }

  n <- length(pp@x)
  ar <- pp@area
  target <- n * (Kfn(pp, R,1)$y)^2 * pi /
    ((ar["xu"] - ar["xl"]) * (ar["yu"] - ar["yl"]))
  if(target == 0) return(0)
  tmp <- pplikfn(1, R, n, pp@x, pp@y, ng, target, F)
  if(tmp <= 0) return(1)
  uniroot(pplikfn, c(0,1), f.lower=-target, f.upper=tmp,
	  R=R, n=n, x=pp@x, y=pp@y, ng=ng, target=target,
	  trace=trace)$root
}

print.pps <- function(x, digits=3) {
  n <- length(x@x)
  cat("point process object of", n, "points\n")
  tmp <- format( t(cbind( signif(x@x, digits), signif(x@y, digits) )) )
    dimnames(tmp) = list(c("x", "y"), rep("", n))
    print(tmp[, 1:min(n, 30)], quote=F)
  if(n > 30) cat("    ....\n")
    invisible(x)
}

setMethod("show", "pps", function(object) print.pps(object))

plot.pps <- function(object, ...)
{
  plot(object@x, object@y, xlab="", ylab="", ...)
}

invisible({
  setInterface("VR_sp_pp2", "C",
               classes = c(rep("single", 2), rep("integer", 2),
                 rep("single", 4)),
               copy = c(rep(F,3), T, rep(F,4)) )
  setInterface("VR_ppset", "C",
               classes = "single", copy = F)
  setInterface("VR_ppget", "C",
               classes = "single", copy = T)
  setInterface("VR_pdata", "C",
               classes = c("integer", "single", "single"),
               copy = c(F,T,T))
  setInterface("VR_simpat", "C",
               classes = c("integer", rep("single", 4), "integer"),
               copy = c(F, T, T, F, F, F) )
  setInterface("VR_simmat", "C",
               classes = c("integer", rep("single", 3)),
               copy = c(F,T,T,F))
  setInterface("VR_plike", "C",
               classes = c(rep("single", 2), "integer", rep("single", 2),
                 "integer", rep("single", 2)),
               copy = c(F,F,F,F,F,F,F,T))
})
