# file nnet/multinom.q copyright (C) 1994-9 W. N. Venables and B. D. Ripley
#
invisible(
setClass("multinom", representation("nnet",
                                    formula = "formula",
                                    weights = "numeric",
                                    deviance = "numeric",
                                    rank = "numeric",
                                    lab = "character",
                                    vcoefnames = "character",
                                    edf = "integer",
                                    AIC = "numeric")
         ))

multinom <- function(formula, data=sys.parent(), weights, subset, na.action,
	    contrasts=NULL, Hess=F, summ=0, censored=F, ...)
{
  class.ind <- function(cl)
  {
    n <- length(cl)
    x <- matrix(0, n, length(levels(cl)))
    x[(1:n) + n * (as.vector(oldUnclass(cl)) - 1)] <- 1
    dimnames(x) <- list(names(cl), levels(cl))
    x
  }
  summ2 <- function(X, Y)
  {
    X <- as.matrix(X)
    Y <- as.matrix(Y)
    n <- nrow(X)
    p <- ncol(X)
    q <- ncol(Y)
    z <- .C("VR_summ2", n,  p, q, Z = t(cbind(X, Y)), na = integer(1))
    Za <- t(z$Z[, 1:z$na, drop = F])
    list(X = Za[, 1:p, drop = F], Y = Za[, p + 1:q])
  }

  call <- match.call()
  m <- match.call(expand = F)
  m$summ <- m$Hess <- m$contrasts <- m$censored <- m$... <- NULL
  m[[1]] <- as.name("model.frame")
  m <- eval(m, sys.parent())
  Terms <- attr(m, "terms")
  X <- model.matrix(Terms, m, contrasts)
  Xr <- qr(X)$rank
  xvars <- as.character(attr(Terms, "variables"))
  if ((yvar <- attr(Terms, "response")) > 0) xvars <- xvars[-yvar]
  xlev <- if (length(xvars) > 0) {
      xlev <- lapply(m[xvars], levels)
      xlev[!sapply(xlev, is.null)]
  }
  Y <- model.extract(m, response)
  if(!is.matrix(Y)) Y <- as.factor(Y)
  w <- model.extract(m, weights)
  if(length(w) == 0)
    if(is.matrix(Y)) w <- rep(1, dim(Y)[1])
    else w <- rep(1, length(Y))
  lev <- levels(Y)
  if(is.factor(Y)) {
    counts <- table(Y)
    if(any(counts == 0)) {
      warning(paste("group(s)", paste(lev[counts == 0], collapse=" "),
		    "are empty"))
      Y <- factor(Y, levels=lev[counts > 0])
      lev <- lev[counts > 0]
    }
    if(length(lev) == 2) Y <- as.vector(oldUnclass(Y)) - 1
    else Y <- class.ind(Y)
  }
  if(summ==1) {
    Z <- cbind(X, Y)
    assign("z1", cumprod(apply(Z, 2, max)+1), frame=1)
    Z1 <- apply(Z, 1, function(x) sum(z1*x))
    oZ <- sort.list(Z1)
    Z2 <- !duplicated(Z1[oZ])
    oX <- (seq(along=Z1)[oZ])[Z2]
    X <- X[oX, , drop=F]
    Y <- if(is.matrix(Y)) Y[oX, , drop=F] else Y[oX]
    w <- diff(c(0,cumsum(w))[c(Z2,T)])
    print(dim(X))
  }
  if(summ==2) {
    Z <- summ2(cbind(X, Y), w)
    X <- Z$X[, 1:ncol(X)]
    Y <- Z$X[, ncol(X) + 1:ncol(Y), drop = F]
    w <- Z$Y
    print(dim(X))
  }
  if(summ==3) {
    Z <- summ2(X, Y*w)
    X <- Z$X
    Y <- Z$Y[, 1:ncol(Y), drop = F]
    w <- rep(1, nrow(X))
    print(dim(X))
  }
  offset <- model.extract(m, offset)
  r <- ncol(X)
  if(is.matrix(Y)) {
# 3 or more response levels or direct matrix spec.
    p <- ncol(Y)
    sY <- Y %*% rep(1, p)
    if(any(sY==0)) stop("some case has no observations")
    if(!censored) {
      Y <- Y / matrix(sY, nrow(Y), p)
      w <- w*sY
    }
    if(length(offset) > 1) {
      if(ncol(offset) !=  p) stop("ncol(offset) is wrong")
      mask <- c(rep(0, r+1+p), rep(c(0, rep(1, r), rep(0, p)), p-1) )
      X <- cbind(X, offset)
      Wts <- as.vector(rbind(matrix(0, r+1, p), diag(p)))
      fit <- nnet1(X, Y, w, Wts=Wts, mask=mask, size=0, skip=T,
                          softmax=T, censored=censored, rang=0, ...)
    } else {
      mask <- c(rep(0, r+1), rep(c(0, rep(1, r)), p-1) )
      fit <- nnet1(X, Y, w, mask=mask, size=0, skip=T, softmax=T,
                          censored=censored, rang=0, ...)
    }
  } else {
# 2 response levels
    if(length(offset) <= 1) {
      mask <- c(0, rep(1, r))
      fit <- nnet1(X, Y, w, mask=mask, size=0, skip=T, entropy=T, rang=0, ...)
    } else {
      mask <- c(0, rep(1, r), 0)
      Wts <- c(rep(0, r+1), 1)
      X <- cbind(X, offset)
      fit <- nnet1(X, Y, w, Wts=Wts, mask=mask, size=0, skip=T,
                  entropy=T, rang=0, ...)
    }
  }
  class(fit) <- "multinom"
  fit@formula <- attr(Terms, "formula")
  fit@call <- call
  fit@weights <- w
  fit@lev <- lev
  fit@deviance <- 2 * fit@value
  fit@rank <- Xr
  edf <- ifelse(length(lev) == 2, 1, length(lev)-1)*Xr
  if(is.matrix(Y)) {
    edf <- (ncol(Y)-1)*Xr
    if(length(dn <- dimnames(Y)[[2]]) > 0) fit@lab <- dn
    else fit@lab <- 1:ncol(Y)
  }
  fit@coefnames <- dimnames(X)[[2]]
  fit@vcoefnames <- fit@coefnames[1:r] # remove offset cols
  attr(fit@vcoefnames, "xlevels") <- xlev
  fit@edf <- edf
  fit@AIC <- fit@deviance + 2 * edf
  if(Hess) {
    mask <- as.logical(mask)
    fit@Hessian <- nnet.Hess(fit, X, Y, w)[mask, mask, drop=F]
    cf <- fit@vcoefnames
    if(length(fit@lev) != 2) {
     bf <- if(length(fit@lev)) fit@lev else fit@lab
     cf <- t(outer(bf[-1], cf, function(x,y) paste(x,y,sep=":")))
    }
    dimnames(fit@Hessian) <- list(cf, cf)
  }
  fit
}

predict.multinom <- function(object, newdata, type=c("class","probs"), ...)
{
  if(!inherits(object, "multinom")) stop("Not a multinom fit")
  type <- match.arg(type)
  if(missing(newdata)) Y <- fitted(object)
  else {
    newdata <- as.data.frame(newdata)
    rn <- row.names(newdata)
    form <- delete.response(terms(object@formula))
    m <- model.frame(form, newdata, na.action = na.omit,
                     xlev = attr(object@vcoefnames, "xlevels"))
    keep <- match(row.names(m), rn)
    X <- model.matrix(form, m)
    Y1 <- predict.nnet(object, X)
    Y <- matrix(NA, nrow(newdata), ncol(Y1),
                dimnames = list(rn, dimnames(Y1)[[2]]))
    Y[keep, ] <- Y1
  }
  switch(type, class={
    if(length(object@lev) > 2)
      Y <- factor(max.col(Y), levels=seq(along=object@lev), labels=object@lev)
    if(length(object@lev) == 2)
      Y <- factor(1 + (Y > 0.5), levels=1:2, labels=object@lev)
    if(length(object@lev) == 0)
      Y <- factor(max.col(Y), levels=seq(along=object@lab), labels=object@lab)
  }, probs={})
  drop(Y)
}

setMethod("show", "multinom", function(object) print.multinom(object))
print.multinom <- function(x, ...)
{
  if(!is.null(cl <- x@call)) {
    cat("Call:\n")
    dput(cl)
  }
  cat("\nCoefficients:\n")
  print(coef(x), ...)
  cat("\nResidual Deviance:", format(x@deviance, nsmall=2), "\n")
  cat("AIC:", format(x@AIC, nsmall=2), "\n")
  invisible(x)
}

coef.multinom <- function(object, ...)
{
  r <- length(object@vcoefnames)
  if(length(object@lev) == 2) {
    coef <- object@wts[1+(1:r)]
    names(coef) <- object@vcoefnames
  } else {
    coef <- matrix(object@wts, nrow = object@n[3], byrow=T)[, 1+(1:r), drop=F]
    if(length(object@lev)) dimnames(coef) <- list(object@lev, object@vcoefnames)
    if(length(object@lab)) dimnames(coef) <- list(object@lab, object@vcoefnames)
    coef <- coef[-1, , drop=F]
  }
  coef
}

drop1.multinom <- function(object, scope, sorted = F, trace = F, ...)
{
  if(!inherits(object, "multinom")) stop("Not a multinom fit")
  if(missing(scope)) scope <- drop.scope(object)
    else {
      if(!is.character(scope))
	scope <- attr(terms(update.formula(object, scope)), "term.labels")
      if(!all(match(scope, attr(object@terms, "term.labels"),
                    nomatch = F)))
	stop("scope is not a subset of term labels")
    }
  ns <- length(scope)
  ans <- matrix(nrow = ns+1, ncol = 2,
                dimnames = list(c("<none>", scope), c("Df", "AIC")))
  ans[1, ] <- c(object@edf, object@AIC)
  i <- 2
  for(tt in scope) {
    cat("trying -", tt,"\n")
    nobject <- update(object, paste("~ . -", tt), trace = trace)
    if(nobject@edf == object@edf) nobject@AIC <- NA
    ans[i, ] <- c(nobject@edf, nobject@AIC)
    i <- i+1
  }
  if(sorted) ans <- ans[sort.list(ans[, 2]), ]
  as.data.frame(ans)
}

add1.multinom <- function(object, scope, sorted = F, trace = F, ...)
{
  if(!inherits(object, "multinom")) stop("Not a multinom fit")
  if(!is.character(scope))
    scope <- add.scope(object, update.formula(object, scope,
					   evaluate = F))
  if(!length(scope))
    stop("no terms in scope for adding to object")
  ns <- length(scope)
  ans <- matrix(nrow = ns+1, ncol = 2,
                dimnames = list(c("<none>",paste("+",scope,sep="")),
                  c("Df", "AIC")))
  ans[1, ] <- c(object@edf, object@AIC)
  i <- 2
  for(tt in scope) {
    cat("trying +", tt,"\n")
    nobject <- update(object, paste("~ . +", tt), trace=trace)
    if(nobject@edf == object@edf) nobject@AIC <- NA
    ans[i, ] <- c(nobject@edf, nobject@AIC)
    i <- i+1
  }
  if(sorted) ans <- ans[sort.list(ans[, 2]), ]
  as.data.frame(ans)
}

extractAIC.multinom <- function(fit, scale, k = 2, ...)
  c(fit@edf, fit@AIC + (k-2)*fit@edf)

vcov.multinom <- function(object)
{
  ginv <- function(X, tol = sqrt(.Machine$double.eps))
    {
    #
    # simplified version of ginv in MASS
    #
      Xsvd <- svd(X)
      Positive <- Xsvd$d > max(tol * Xsvd$d[1], 0)
      if(!any(Positive)) array(0, dim(X)[2:1])
      else Xsvd$v[, Positive] %*% ((1/Xsvd$d[Positive]) * t(Xsvd$u[, Positive]))
    }

  if(is.null(object@Hessian)) {
    cat("\nRe-fitting to get Hessian\n\n")
    Call <- object@call
    Call$Hess <- T; Call$trace <- F
    object <- eval(Call, sys.parent())
  }
  structure(ginv(object@Hessian), dimnames = dimnames(object@Hessian))
}

summary.multinom <-
function(object, correlation = T, digits = options()$digits,
         Wald.ratios = F, ...)
{
  vc <- vcov(object)
  r <- length(object@vcoefnames)
  se <- sqrt(diag(vc))
  if(length(object@lev) == 2) {
    coef <- object@wts[1 + (1:r)]
    stderr <- se
    names(coef) <- names(stderr) <- object@vcoefnames
  } else {
    coef <- matrix(object@wts, nrow = object@n[3],
		   byrow = T)[-1, 1 + (1:r), drop = F]
    stderr <- matrix(se, nrow = object@n[3] - 1, byrow = T)
    if(length(l <- object@lab) || length(l <- object@lev))
      dimnames(coef) <- dimnames(stderr) <- list(l[-1], object@vcoefnames)
  }
  object <- as.list(object)
  object$is.binomial <- (length(object@lev) == 2)
  object$digits <- digits
  object$coefficients <- coef
  object$standard.errors <- stderr
  if(Wald.ratios) object$Wald.ratios <- coef/stderr
  if(correlation) object$correlation <- vc/outer(se, se)
  oldClass(object) <- "summary.multinom"
  object
}

print.summary.multinom <- function(x, digits = x$digits, ...)
{
  if(!is.null(cl <- x$call)) {
    cat("Call:\n")
    dput(cl)
  }
  cat("\nCoefficients:\n")
  if(x$is.binomial) {
    print(cbind(Values = x$coefficients,
		"Std. Err." = x$standard.errors,
		"Value/SE" = x$Wald.ratios),
	  digits = digits)
  } else {
    print(x$coefficients, digits = digits)
    cat("\nStd. Errors:\n")
    print(x$standard.errors, digits = digits)
    if(!is.null(x$Wald.ratios)) {
      cat("\nValue/SE (Wald statistics):\n")
      print(x$coefficients/x$standard.errors, digits = digits)
    }
  }
  cat("\nResidual Deviance:", format(x$deviance, nsmall=2), "\n")
  cat("AIC:", format(x$AIC, nsmall=2), "\n")
  if(!is.null(correl <- x$correlation)) {
    p <- dim(correl)[2]
    if(p > 1) {
      cat("\nCorrelation of Coefficients:\n")
      ll <- lower.tri(correl)
      correl[ll] <- format(round(correl[ll], digits))
      correl[!ll] <- ""
      print(correl[-1, -p], quote = F, ...)
    }
  }
  invisible(x)
}

anova.multinom <- function(object, ..., test = c("Chisq", "none"))
{
  test <- match.arg(test)
  dots <- list(...)
  if(length(dots) == 0)
    stop("anova is not implemented for a single multinom object")
  mlist <- list(object, ...)
  nt <- length(mlist)
  dflis <- sapply(mlist, function(x) x@edf)
  s <- sort.list(dflis)
  dflis <- nrow(residuals(object)) * (ncol(residuals(object))-1) - dflis
  mlist <- mlist[s]
  if(any(!sapply(mlist, inherits, "multinom")))
    stop("not all objects are of class `multinom'")
  rsp <- unique(sapply(mlist, function(x) paste(formula(x)[2])))
  mds <- sapply(mlist, function(x) paste(formula(x)[3]))
  dfs <- dflis[s]
  lls <- sapply(mlist, function(x) deviance(x))
  tss <- c("", paste(1:(nt - 1), 2:nt, sep = " vs "))
  df <- c(NA, -diff(dfs))
  x2 <- c(NA, -diff(lls))
  pr <- c(NA, 1 - pchisq(x2[-1], df[-1]))
  out <- data.frame(Model = mds, Resid.df = dfs,
                    Deviance = lls, Test = tss, Df = df, LRtest = x2,
                    Prob = pr)
  names(out) <- c("Model", "Resid. df", "Resid. Dev", "Test",
                  "   Df", "LR stat.", "Pr(Chi)")
  if(test=="none") out <- out[, 1:6]
  oldClass(out) <- c("anova", "data.frame")
  attr(out, "heading") <-
    c("Likelihood ratio tests of Multinomial Models\n",
      paste("Response:", rsp))
  out
}

formula.multinom <- function(object) object@call[[2]]
fitted.multinom <- function(object, ...) object@fitted
residuals.multinom <- function(object, ...) object@residuals
deviance.multinom <- function(object, ...) object@deviance

model.frame.multinom <-
function(formula, data = NULL, na.action = NULL, ...)
{
    oc <- formula@call
    oc[[1]] <- as.name("model.frame")
    m <- match(names(oc)[-1], c("formula", "data", "na.action", "subset"))
    oc <- oc[c(T, !is.na(m))]
    if(length(data)) {
        oc$data <- substitute(data)
        eval(oc, sys.parent())
    }
    else eval(oc, list())
}

extractAIC.multinom <- function(fit, scale, k = 2, ...)
  c(fit@edf, fit@AIC + (k-2)*fit@edf)

update.multinom <- function(object, formula, ..., evaluate = T, class)
{
    if(is.null(newcall <- object@call))
        stop("need an object with call component or attribute")
    tempcall <- match.call(expand.dots = F)$... #will be list(...)
    if(!missing(formula))
        newcall$formula <- as.vector(update.formula(object@formula, formula, evaluate = T))
    else {
        nc <- names(sys.call())
        if(length(nc) && any(pmatch(nc, "formula", 0)))
            newcall$formula <- NULL
    }
    if(!missing(class) && is.character(class))
        newcall[[1]] <- as.name(class[1])
    if(length(tempcall) > 1) {
        def <- getFunction(newcall[[1]])
        def$formula <- NULL
        TT <- match.call(def, tempcall)
        if((ndots <- length(TT)) < length(tempcall)) {
            nt <- pmatch(names(tempcall), names(def)[ - length(def)])
            nt <- names(def)[nt]
            nT <- names(TT)
            for(i in nt[is.na(match(nt, nT))])
                newcall[[i]] <- NULL
        }
        if(ndots > 1) {
            ndots <- names(TT)[-1]
            newcall[ndots] <- TT[ndots]
        }
    }
    if(evaluate)
        eval(newcall, sys.parent())
    else newcall
}
