addMassMenus <- function()
{
    if(!exists("guiCreate")) stop("Not running under the GUI")
    stat.loc <-
        guiGetPropertyValue("MenuItem",
                            Name=paste(guiGetMenuBar(),"Statistics", sep="$"),
                            PropName="Index")
    guiCreate("MenuItem",
              Name=paste(guiGetMenuBar(),"MASS", sep="$"), Type="Menu",
              MenuItemText= "&MASS", Index=as.numeric(stat.loc) + 1,
              OverWrite=F)
    guiCreate( "MenuItem",
              Name=paste(guiGetMenuBar(),"MASS", "loglm", sep="$"),
              Type="MenuItem", Action="Function", Command="menuLoglm",
              MenuItemText= "Log-linear Models...")
    guiCreate( "MenuItem",
              Name=paste(guiGetMenuBar(),"MASS", "GLM_NB", sep="$"),
              Type="MenuItem", Action="Function", Command="menuGlmnb",
              MenuItemText= "Negative Binomial GLM...")
    guiCreate("MenuItem",
              Name=paste(guiGetMenuBar(),"MASS", "StepAIC", sep="$"),
              Type="MenuItem", Action="Function", Command="menuStepAIC",
              MenuItemText= "StepAIC...")
    guiCreate("MenuItem",
              Name=paste(guiGetMenuBar(),"MASS", "LDA", sep="$"),
              Type="MenuItem", Action="Function", Command="menuLDA",
              MenuItemText= "Linear Discriminant Analysis...")
    guiCreate("MenuItem",
              Name=paste(guiGetMenuBar(),"MASS", "QDA", sep="$"),
              Type="MenuItem", Action="Function", Command="menuQDA",
              MenuItemText= "Quadratic Discriminant Analysis...")
    guiCreate( "MenuItem",
              Name=paste(guiGetMenuBar(),"MASS", "MDS", sep="$"),
              Type="MenuItem", Action="Function", Command="menuMDS",
              MenuItemText= "MultiDimensional Scaling...")
    createLDAmenu()
    createQDAmenu()
    createStepMenu()
    createGlmnbMenu()
    createLLmenu()

    lib <- searchPaths()[match("MASS", search())]
    guiLoadDefaultObjects("Property",
                          FileName = paste(lib, "MASSprop.dft", sep = "/"))
    guiLoadDefaultObjects("FunctionInfo",
                          FileName = paste(lib, "MASSfun.dft", sep = "/"))
    cat("MASS menus and dialogs added\n")
}

removeMassMenus <- function()
{
    if(!exists("guiRemove")) stop("Not running under the GUI")
    removeLLmenu()
    removeGlmnbMenu()
    removeStepMenu()
    removeQDAmenu()
    removeLDAmenu()
    item.names <- guiGetObjectNames("MenuItem")
    mass.items <- item.names[grep("\\$MASS\\$", item.names)]
    for (i in mass.items) guiRemove("MenuItem", name = i)
    guiRemove("MenuItem", name = item.names[grep("\\$MASS$", item.names)])
    cat("MASS menus and dialogs removed\n")
}

# ========== LDA ============

createLDAmenu <- function()
{
    if(!exists("guiCreate")) return(NULL)
    # Context menus
    guiCreate("ClassInfo", Name="lda", ContextMenu="lda",
              DoubleClickAction = "tabResults.lda")
    guiCreate("MenuItem", Name="lda", Type="Menu",
              DocumentType="lda")
    guiCreate("MenuItem", Name="lda$Summary", Type="MenuItem",
              DocumentType="lda", Action="Function",
              Command="tabResults.lda", MenuItemText="Summary",
              ShowDialogOnRun=F)
    guiCreate("MenuItem", Name="lda$Plot", Type="MenuItem",
              DocumentType="lda", Action="Function",
              Command="tabPlot.lda", MenuItemText="Plot...",
              ShowDialogOnRun=T)
    guiCreate("MenuItem", Name="lda$Predict", Type="MenuItem",
              DocumentType="lda", Action="Function",
              Command="tabPredict.lda", MenuItemText="Predict...",
              ShowDialogOnRun=T)
    invisible()
}

removeLDAmenu <- function()
{
    if(!exists("guiRemove")) return(NULL)
    guiRemove("MenuItem", Name="lda")
    guiRemove("ClassInfo", Name="lda")
    invisible()
}


backLDA <- function(data)
{
    data <- backFormMASS(data)
    activeprop <- cbGetActiveProp(data)
    activevalue <- cbGetCurrValue(data, activeprop)
    switch(activeprop,
           ldaPlot = {
               data <- cbSetEnableFlag(data, "ldaPlotDimen", activevalue=="T")
           }
           ,
           ldaFitMethod = {
               if(activevalue == "moment" || activevalue == "mle") {
                   data <- cbSetEnableFlag(data, "ldaCV", T)
               } else {
                   data <- cbSetEnableFlag(data, "ldaCV", F)
                   data <- cbSetCurrValue(data, "ldaCV", "F")
               }
               data <- cbSetEnableFlag(data, "ldaFitNu", activevalue=="t")
           }
           ,
           ldaCV = {
               set <- activevalue == "F"
               data <- cbSetEnableFlag(data, "ldaPlot", set)
               data <- cbSetEnableFlag(data, "ldaPredictMethod", set)
               data <- cbSetEnableFlag(data, "SPropPredictNewdata", set)
               data <- cbSetEnableFlag(data, "SPropSavePredictObject", set)
           } ,)
    data
}


menuLDA <- function(x, y, subset, na.omit.p=T,
                    method="moment", newdata=NULL, predict.save=NULL,
                    predict.method="plug-in", plot.p=F, plot.dimen=99,
                    CV=F, nu=5)
{
    fun.call <- match.call()
    fun.call[[1]] <- as.name("lda.formula")
    if(na.omit.p) fun.call$na.action <- as.name("na.omit")
    else fun.call$na.action <- as.name("na.fail")
    fun.args <- is.element(arg.names(fun.call),
                           c("x", "y", "subset", "na.action",
                             "method", "CV", "nu"))
    fun.call <- fun.call[c(T, fun.args)]
    ldaobj <- eval(fun.call)
    if(!CV) tabResults.lda(ldaobj)
    if(!CV) tabPredict.lda(ldaobj, newdata, predict.save, predict.method)
    if(!CV && plot.p) tabPlot.lda(ldaobj, plot.dimen=plot.dimen)
    invisible(ldaobj)
}

tabResults.lda <- function(object)
{
    cat("\n\t*** Linear Discriminant Analysis ***\n")
    invisible(print.lda(object))
}

tabPredict.lda <- function(object, newdata=NULL, predict.save=NULL,
                           predict.method="plug-in")
{
    if(!is.null(predict.save)) {
        if(is.null(newdata))
            predobj <- predict(object, method=predict.method)
        else
            predobj <- predict(object, newdata=newdata, method=predict.method)
        if(exists(predict.save, where = 1)) {
            newsave.name <- unique.name(predict.save, where = 1)
            assign(newsave.name, predobj, where = 1)
            warning(paste("Predictions saved in", newsave.name))
        } else assign(predict.save, predobj, where = 1)
    }
    invisible(NULL)
}

tabPlot.lda <- function(object, plot.dimen=99)
{
    plot.lda(object, dimen=plot.dimen)
    invisible(NULL)
}

# ========== QDA ============

backQDA <- function(data)
{
    data <- backFormMASS(data)
    activeprop <- cbGetActiveProp(data)
    activevalue <- cbGetCurrValue(data, activeprop)
    switch(activeprop,
           qdaFitMethod = {
               if(activevalue == "moment" || activevalue == "mle") {
                   data <- cbSetEnableFlag(data, "qdaCV", T)
               } else {
                   data <- cbSetEnableFlag(data, "qdaCV", F)
                   data <- cbSetCurrValue(data, "qdaCV", "F")
               }
               data <- cbSetEnableFlag(data, "qdaFitNu", activevalue=="t")
           }
           ,
           qdaCV = {
               set <- activevalue == "F"
               data <- cbSetEnableFlag(data, "qdaPredictMethod", set)
               data <- cbSetEnableFlag(data, "SPropPredictNewdata", set)
               data <- cbSetEnableFlag(data, "SPropSavePredictObject", set)
           } ,)
    data
}

menuQDA <- function(x, y, subset, na.omit.p=T,
                    method="moment", newdata=NULL, predict.save=NULL,
                    predict.method="plug-in", CV=F, nu=5)
{
    fun.call <- match.call()
    fun.call[[1]] <- as.name("qda.formula")
    if(na.omit.p) fun.call$na.action <- as.name("na.omit")
    else fun.call$na.action <- as.name("na.fail")
    fun.args <- is.element(arg.names(fun.call),
                           c("x", "y", "subset", "na.action",
                             "method", "CV", "nu"))
    fun.call <- fun.call[c(T, fun.args)]
    qdaobj <- eval(fun.call)
    if(!CV) tabPredict.qda(qdaobj, newdata, predict.save, predict.method)
    invisible(qdaobj)
}

tabPredict.qda <- function(object, newdata=NULL, predict.save=NULL,
                           predict.method="plug-in")
{
    if(!is.null(predict.save)) {
        if(is.null(newdata))
            predobj <- predict(object, method=predict.method)
        else
            predobj <- predict(object, newdata=newdata, method=predict.method)
        if(exists(predict.save, where = 1)) {
            newsave.name <- unique.name(predict.save, where = 1)
            assign(newsave.name, predobj, where = 1)
            warning(paste("Predictions saved in", newsave.name))
        } else assign(predict.save, predobj, where = 1)
    }
    invisible(NULL)
}

tabResults.qda <- function(object)
{
    cat("\n\t*** Quadratic Discriminant Analysis ***\n")
    invisible(print.qda(object))
}

createQDAmenu <- function()
{
    if(!exists("guiCreate")) return(NULL)

    # Context menus
    guiCreate("ClassInfo", Name="qda", ContextMenu="qda",
              DoubleClickAction = "tabResults.qda")
    guiCreate("MenuItem", Name="qda", Type="Menu",
              DocumentType="qda")
    guiCreate("MenuItem", Name="qda$Summary", Type="MenuItem",
              DocumentType="qda", Action="Function",
              Command="tabResults.qda", MenuItemText="Summary",
              ShowDialogOnRun=F)
    guiCreate("MenuItem", Name="qda$Predict", Type="MenuItem",
              DocumentType="qda", Action="Function",
              Command="tabPredict.qda", MenuItemText="Predict...",
              ShowDialogOnRun=T)
    invisible()
}

removeQDAmenu <- function()
{
    if(!exists("guiRemove")) return(NULL)
    guiRemove("MenuItem", Name="qda")
    guiRemove("ClassInfo", Name="qda")
    invisible()
}

# =========== stepAIC ==========

menuStepAIC <-
function(object, formula.upper, formula.lower = NULL,
         direction = "both", trace.p = T, print.short.p = F,
         print.long.p = T, print.anova.p = F, print.correlation.p = F)
{
    if(trace.p)  cat("\n\t*** Stepwise Fitting process ***\n")
    stepobj <- stepAIC(object, scope = list(upper = formula.upper,
                               lower = formula.lower),
                       direction = direction, trace = trace.p)
    if(print.short.p || print.long.p || print.anova.p) {
        cat("\n\t*** Chosen Fit ***\n")
        if(print.short.p) print(stepobj)
        if(print.long.p)
            print(summary(stepobj, correlation = print.correlation.p))
        if(print.anova.p) {
            cat("\n\t*** Stepwise ANOVA ***\n")
            print(stepobj$anova)
        }
        cat("\n")
    }
    invisible(stepobj)
}

backStepAIC <- function(data)
{
    activeprop <- cbGetActiveProp(data)
    if(activeprop == "stepInitObj") {
        if(exists(initobj <- cbGetCurrValue(data, "stepInitObj"))) {
            form <-
                deparse(as.vector(eval(parse(text=paste("formula(",initobj,")")))))
            form <- paste(form, collapse=" ")
            data <- cbSetCurrValue(data, "SPropPFFormula", form)
        }
    }
    else if(activeprop == "SPropPFButton") {
        if(exists(initobj <- cbGetCurrValue(data, "stepInitObj"))) {
            guiModify(classname = "Property", Name = "SPropCFFormulaID",
                      DialogControl = "Invisible",
                      DefaultValue = cbGetDialogId(data))
            df <- eval(parse(text=paste(initobj, "$call$data", sep="")))
            guiModify(classname = "Property", Name = "SPropCFData",
                      DialogControl = "Invisible", DefaultValue = df)
            form <- deparse(as.vector(eval(parse(text=paste("formula(",initobj,")")))))
            form <- paste(form, collapse=" ")
            guiModify(classname = "Property", Name = "SPropCFFormula",
                      DefaultValue = form)
            guiDisplayDialog(classname = "Function", Name = "makeFormulaStepAIC")
            data <- cbSetEnableFlag(data, "SPropDataFrameList", F)
            data <- cbSetEnableFlag(data, "SPropPFFormula", F)
            data <- cbSetEnableFlag(data, "SPropPFFormula2", F)
            data <- cbSetEnableFlag(data, "SPropPFButton", F)
        }
        else guiCreate("MessageBox", String =
                       "Enter initial object name before building formula.")
    }
    else if(activeprop == "SPropPrintShort") {
        data <- cbSetEnableFlag(data, "SPropPrintCor", F)
        data <- cbSetCurrValue(data, "SPropPrintLong", "F")
    }
    else if(activeprop == "SPropPrintLong") {
        data <- cbSetEnableFlag(data, "SPropPrintCor", T)
        data <- cbSetCurrValue(data, "SPropPrintShort", "F")
    }
    else if(activeprop == "SPropPFEnableButton") {
        data <- cbSetEnableFlag(data, "SPropDataFrameList", T)
        data <- cbSetEnableFlag(data, "SPropPFFormula", T)
        data <- cbSetEnableFlag(data, "SPropPFFormula2", T)
        data <- cbSetEnableFlag(data, "SPropPFButton", T)
    }
    data
}

createStepMenu <- function()
{
    if(!exists("guiCreate")) return(NULL)
    makeFormulaStepAIC <-
        function(data, formula = NULL, formula.id, ...)
            invisible(NULL)

    guiCreate( "FunctionInfo", Name = "makeFormulaStepAIC",
              Function = "makeFormulaStepAIC",
              DialogHeader = "Formula",
              PropertyList = "SPropDummy0,
SPropCFData, SPropCFFormulaID, SPropPTVariables, SPropPTVarSelect,
SPropPTEnableButton, SPropFVarG, SPropFTermsButtonG, SPropCFFormula,
SPropStepWhichF, SPropFTermsListG, SPropFRemoveG, SPropFSpace1,
SPropFSpace2, SPropFSpace3",
              ArgumentList = "#0=SPropDummy0,#1=SPropCFData,
               #2=SPropCFFormula,#3=SPropCFFormulaID",
              CallBackFunction = "backFormulaStep")
    invisible()
}

removeStepMenu <- function()
{
    guiRemove("FunctionInfo", Name="makeFormulaStepAIC")
    invisible()
}

# =========== glm.nb =============
menuGlmnb <-
function(formula, link = identity, data, weights, subset, na.omit.p = T,
         trace = F,
	maxit = 10, epsilon = 0.001, print.short.p = F,
	print.long.p, print.anova.p = T, print.correlation.p = F,
	save.name = NULL, save.fit.p = F, save.resid.working.p = F,
	save.resid.pearson.p = F, save.resid.deviance.p = F,
	save.resid.response.p = F, plotResidVsFit.p = F,
	plotSqrtAbsResid.p = F, plotResponseVsFit.p = F, plotQQ.p
	 = F, smooths.p = F, rugplot.p = F, id.n = 3,
	plotPartialResid.p = F, plotPartialFit.p = F,
	rugplotPartialResid.p = F, scalePartialResid.p = T, newdata
	 = NULL, predobj.name = NULL, predict.type = "link",
	predict.p = F, se.p = F)
{
    fun.call <- match.call()
    fun.call[[1]] <- as.name("glm.nb")
    if(na.omit.p) fun.call$na.action <- as.name("na.omit")
    else fun.call$na.action <- as.name("na.fail")
    control.list <- list(epsilon = epsilon, maxit = maxit, trace = trace)
    fun.call[["control"]] <- control.list
    fun.args <- is.element(arg.names(fun.call), arg.names("glm.nb"))
    fun.call <- fun.call[c(T, fun.args)]
    glmobj <- eval(fun.call)
    #
    # Call summary function:
    tabSummary.glm(glmobj, print.short.p, print.long.p,
                   print.correlation.p, print.anova.p, save.name,
                   save.fit.p, save.resid.working.p,
                   save.resid.pearson.p, save.resid.deviance.p,
                   save.resid.response.p)
    #
    # Call plot function
    if(any(c(plotResidVsFit.p, plotSqrtAbsResid.p,
             plotResponseVsFit.p, plotQQ.p, plotPartialResid.p))
       ) tabPlot.glm(glmobj, plotResidVsFit.p,
                     plotSqrtAbsResid.p, plotResponseVsFit.p,
                     plotQQ.p, smooths.p, rugplot.p, id.n,
                     plotPartialResid.p, plotPartialFit.p,
                     rugplotPartialResid.p, scalePartialResid.p)
    #
    # Call predict:
    if(any(c(predict.p, se.p)))
        tabPredict.glm(glmobj, newdata, predobj.name,
                       predict.type, predict.p, se.p)
    invisible(glmobj)
}


createGlmnbMenu <- function()
{
    if(!exists("guiCreate")) return(NULL)

    # Context menu (as inheritance does not work)
    guiCreate("ClassInfo", Name="negbin", ContextMenu="negbin")

    guiCreate( "MenuItem", Name = "negbin",
              Type = "Menu", DocumentType = "negbin")

    guiCreate( "MenuItem", Name = "negbin$Summary",
              Type = "MenuItem", DocumentType = "negbin",
              MenuItemText = "Summary...", Action = "Function",
              Command = "tabSummary.glm", ShowDialogOnRun = T)

    guiCreate( "MenuItem", Name = "negbin$Plot",
              Type = "MenuItem", DocumentType = "negbin",
              MenuItemText = "Plot...", Action = "Function",
              Command = "tabPlot.glm", ShowDialogOnRun = T)

    guiCreate( "MenuItem", Name = "negbin$Predict",
              Type = "MenuItem", DocumentType = "negbin",
              MenuItemText = "Predict...", Action = "Function",
              Command = "tabPredict.glm", ShowDialogOnRun = T)
    invisible()
}

removeGlmnbMenu <- function()
{
    if(!exists("guiRemove")) return(NULL)
    guiRemove( "ClassInfo", Name = "negbin")
    guiRemove( "MenuItem", Name = "negbin")
}

#======== MDS =========

menuMDS <-
function(x, subset, na.omit.p=T, dist.func="dist",
         metric="euclidean", d.scale="raw", type="cmdscale", k=2,
         trace=F, niter=50, magic=0.2, tol=0.0001, add=F, plot.it=F,
         stand = F, ordratio.list, logratio.list, asymm.list)
{
    fun.call <- match.call()
    if(!missing(subset)) x <- x[subset, , drop=F]
    if(na.omit.p) x <- na.omit(x)
    if(dist.func=="dist") {
#   variable scaling ="raw, 0-1, unit variance, PCA",
        if(d.scale == "0-1") x <- sapply(x, function(x){
            r <- range(x); (x-r[1])/(r[2]-r[1])})
        if(d.scale == "unit variance") x <- scale(x)
        if(d.scale == "PCA") {
            x.pca <- princomp(x)
            x <- scale(predict(x.pca), F, x.pca$sdev)
        }
        x <- as.matrix(x)
        distobj <- do.call(dist.func, list(x=x, metric=metric))
    }
    if(dist.func=="daisy") {
        library(cluster)
        typelist <- NULL
        if(!missing(ordratio.list) || !missing(logratio.list) ||
           ! missing(asymm.list)) {
            type.list <- list()
            if(!missing(ordratio.list))
                type.list$ordratio <- unlist(ordratio.list)
            if(!missing(logratio.list))
                type.list$logratio <- unlist(logratio.list)
            if(!missing(asymm.list)) type.list$asymm <- unlist(asymm.list)
            typelist <- as.name(deparse(type.list))
        }
        distobj <- do.call(dist.func, list(x=x, metric=metric,
                                           stand=stand, type=typelist))
    }
    fun.args <- is.element(arg.names(fun.call), arg.names(type))
    m.call <- fun.call[c(T, fun.args)]
    m.call[[1]] <- as.name(type)
    m.call$d <- as.name("distobj")
    if(type=="isoMDS") m.call$maxit <- niter
    if(type=="isoMDS" || type=="sammon")
        m.call$y <- cmdscale(distobj, k=k)
    object <- eval(m.call)
    if(plot.it) {
        if(is.list(object)) x <- object$points else x <- object
        if(k==2) eqscplot(x, xlab="", ylab="")
        if(k==3) spin(x)
    }
    invisible(object)
}

backMDS <- function(data)
{
    activeprop <- cbGetActiveProp(data)
    activevalue <- cbGetCurrValue(data, activeprop)
    if(cbIsInitDialogMessage(data)) {
        data <- cbSetEnableFlag(data, "mdsCmdAdd", T)
        data <- cbSetEnableFlag(data, "mdsCmdMagic", F)
        data <- cbSetEnableFlag(data, "mdsCmdTol", F)
        data <- cbSetEnableFlag(data, "mdsCmdTrace", F)
        data <- cbSetEnableFlag(data, "mdsCmdNiter", F)
        data <- cbSetEnableFlag(data, "mdsClusterOrdinalRatio", F)
        data <- cbSetEnableFlag(data, "mdsClusterLogRatio", F)
        data <- cbSetEnableFlag(data, "mdsClusterAsymmetricBinary", F)
        data <- cbSetEnableFlag(data, "mdsClusterDissStand", F)
        data <- cbSetCurrValue(data, "SPropOmitMissing", "F")
    }
    switch(activeprop,
           mdsType = {
               data <- cbSetEnableFlag(data, "mdsCmdAdd",
                                       activevalue=="cmdscale")
               data <- cbSetEnableFlag(data, "mdsCmdMagic",
                                       activevalue=="sammon")
               data <- cbSetEnableFlag(data, "mdsCmdTol",
                                       activevalue=="isoMDS")
               data <- cbSetEnableFlag(data, "mdsCmdTrace",
                                       activevalue!="cmdscale")
               data <- cbSetEnableFlag(data, "mdsCmdNiter",
                                       activevalue!="cmdscale")
           },
           mdsDist = {
               if(activevalue=="dist") {
                   data <- cbSetEnableFlag(data, "mdsClusterOrdinalRatio", F)
                   data <- cbSetEnableFlag(data, "mdsClusterLogRatio", F)
                   data <- cbSetEnableFlag(data, "mdsClusterAsymmetricBinary", F)
                   data <- cbSetEnableFlag(data, "mdsClusterDissStand", F)
                   data <- cbSetEnableFlag(data, "mdsDistScale", T)
               }
               if(activevalue=="daisy") {
                   data <- cbSetEnableFlag(data, "mdsClusterOrdinalRatio", T)
                   data <- cbSetEnableFlag(data, "mdsClusterLogRatio", T)
                   data <- cbSetEnableFlag(data, "mdsClusterAsymmetricBinary", T)
                   data <- cbSetEnableFlag(data, "mdsClusterDissStand", T)
                   data <- cbSetEnableFlag(data, "mdsDistScale", F)
               }
           },
           mdsDimen = {
               if(activevalue=="2" || activevalue=="3") {
                   data <- cbSetEnableFlag(data, "mdsPlot", T)
               } else {
                   data <- cbSetEnableFlag(data, "mdsPlot", F)
                   data <- cbSetCurrValue(data, "mdsPlot", "F")
               }
           }
           )
    data
}

#======= Loglm =========

createLLmenu <- function()
{
    if(!exists("guiCreate")) return(NULL)
    # Context menus
    guiCreate("ClassInfo", Name="loglm", ContextMenu="loglm")
    guiCreate("MenuItem", Name="loglm", Type="Menu",
              DocumentType="loglm")
    guiCreate("MenuItem", Name="loglm$Summary", Type="MenuItem",
              DocumentType="loglm", Action="Function",
              Command="tabSummary.loglm", MenuItemText="Summary",
              ShowDialogOnRun=F)

    invisible()
}

removeLLmenu <- function()
{
    if(!exists("guiRemove")) return(NULL)
    guiRemove("ClassInfo", Name="loglm")
    guiRemove("MenuItem", Name="loglm")
    invisible()
}

tabSummary.loglm <- function(object, p.short=F, p.long=T, fitted.p=T)
{
    if(p.short || p.long) {
        cat("\n\t*** Log-Linear Model ***\n")
        if(p.short) print(object)
        if(p.long) print(summary(object, fitted=fitted.p))
    }
    invisible()
}

menuLoglm <- function(formula, data=NULL, weights, subset, na.omit.p=T,
                      iter=20, eps=0.01, p.short=F, p.long=T, fitted.p=T)
{
    if(is.null(data)) {
        display.messagebox("Data argument is needed for loglm")
        #    stop("Data argument is needed for loglm")
        return(NULL)
    }
    fun.call <- match.call()
    fun.call[[1]] <- as.name("loglm")
    if(na.omit.p) fun.call$na.action <- as.name("na.omit")
    else fun.call$na.action <- as.name("na.fail")
    fun.args <- is.element(arg.names(fun.call),
                           c(arg.names("loglm"), "iter", "eps"))
    fun.call <- fun.call[c(T, fun.args)]
    if(p.long && fitted.p) {
        fun.call$fit <- T
        fun.call$keep.frequencies <- T
    }
    loglmobj <- eval(fun.call)
    tabSummary.loglm(loglmobj, p.short, p.long, fitted.p)
    invisible(loglmobj)
}


backLoglm <- function(data)
{
    LLfindcols <- function(data, data.name, var.name)
    {
        if(exists(cbGetCurrValue(data, data.name))) {
            obj <- eval(as.name(cbGetCurrValue(data, data.name)))
            if(!is.data.frame(obj) && is.array(obj))
                variables.x <- names(dimnames(obj))
            else variables.x <- names(obj)
            ncol <- length(variables.x)
            if(ncol > 0) {
                list.x <- as.name(variables.x[1])
                if(ncol >= 2)
                    for(i in 2:ncol)
                        list.x <- paste(list.x, as.name(variables.x[i]), sep = ",")
            } else list.x <- NULL
            data <- cbSetOptionList(data, var.name, list.x)
        }
        data
    }
    activeprop <- cbGetActiveProp(data)
    activevalue <- cbGetCurrValue(data, activeprop)
    if(cbIsInitDialogMessage(data)) {
        data <- cbSetCurrValue(data, "SPropMaxiter", 20)
        data <- cbSetCurrValue(data, "SPropTolerance", 0.1)
        if(exists(cbGetCurrValue(data, "SPropDataFrameList"))) {
            data <- LLfindcols(data, "SPropDataFrameList", "SPropMVVariables")
            list.x <- paste("<ALL>",
                            cbGetOptionList(data, "SPropMVVariables"), sep = ",")
            data <- cbSetOptionList(data, "SPropMVVariables", list.x)
        }
    }
    else if(activeprop == "SPropPrintShort") {
        data <- cbSetEnableFlag(data, "loglmFitted", F)
        data <- cbSetCurrValue(data, "SPropPrintLong", "F")
    }
    else if(activeprop == "SPropPrintLong") {
        data <- cbSetEnableFlag(data, "loglmFitted", T)
        data <- cbSetCurrValue(data, "SPropPrintShort", "F")
    }
    else if(cbGetActiveProp(data) == "SPropDataFrameList" &&
            !cbIsCancelMessage(data)) {
        if(exists(cbGetCurrValue(data, "SPropDataFrameList"))) {
            data <- LLfindcols(data, "SPropDataFrameList", "SPropMVVariables")
            list.x <- paste("<ALL>",
                            cbGetOptionList(data, "SPropMVVariables"), sep = ",")
            data <- cbSetOptionList(data, "SPropMVVariables", list.x)
        }
    }
    else if(cbGetActiveProp(data) == "SPropMVVariables") {
        curr.formula <- cbGetCurrValue(data, "loglmFormulaNR")
        if(nchar(curr.formula) == 0) curr.formula <- "~"
        curr.formula <- unlist(unpaste(curr.formula, sep = "~"))
        curr.var <- cbGetCurrValue(data, "SPropMVVariables")
        if(nchar(curr.var) != 0)
            mvar <- unlist(unpaste(curr.var, sep = ","))
        else mvar <- curr.var
        if(is.na(match("<ALL>", mvar)))
            if(length(mvar) >= 2) curr.var <- paste(mvar, collapse = "+")
            else curr.var <- mvar
        else {
            obj <- eval(as.name(cbGetCurrValue(data, "SPropDataFrameList")))
            if(!is.data.frame(obj) && is.array(obj))
                variables.x <- names(dimnames(obj))
            else variables.x <- names(obj)
            curr.var <- paste(variables.x, collapse = "+")
        }
        curr.formula[[2]] <- curr.var
        curr.formula <- paste(curr.formula, collapse = "~")
        data <- cbSetCurrValue(data, "loglmFormulaNR", curr.formula)
    }
    data
}

backSummLoglm <- function(data)
{
    activeprop <- cbGetActiveProp(data)
    if(activeprop == "SPropPrintShort") {
        data <- cbSetEnableFlag(data, "loglmFitted", F)
        data <- cbSetCurrValue(data, "SPropPrintLong", "F")
    }
    else if(activeprop == "SPropPrintLong") {
        data <- cbSetEnableFlag(data, "loglmFitted", T)
        data <- cbSetCurrValue(data, "SPropPrintShort", "F")
    }
    data
}


backFormMASS <- function(data)
{
    initialmsg <- cbIsInitDialogMessage(data)
    activeprop <- cbGetActiveProp(data)
    activevalue <- cbGetCurrValue(data, activeprop)
    if(initialmsg) {
        if(exists(cbGetCurrValue(data, "SPropDataFrameList"))) {
            data <- spropColName(data, "SPropDataFrameList", "SPropDependent")
            list.x <- paste("<ALL>",
                            cbGetOptionList(data, "SPropDependent"), sep = ",")
            data <- cbSetOptionList(data, "SPropIndependent", list.x)
        }
        data <- cbSetCurrValue(data, "SPropDependent", "")
        data <- cbSetCurrValue(data, "SPropIndependent", "")
    }
    switch(activeprop,
           SPropDataFrameList = {
               if(exists(cbGetCurrValue(data, "SPropDataFrameList"))) {
                   data <- spropColName(data, "SPropDataFrameList",
                                        "SPropDependent")
                   list.x <- paste("<ALL>",
                                   cbGetOptionList(data, "SPropDependent"),
                                   sep = ",")
                   data <- cbSetOptionList(data, "SPropIndependent", list.x)
               }
               data <- cbSetCurrValue(data, "SPropDependent", "")
               data <- cbSetCurrValue(data, "SPropIndependent", "")
           }
           ,
           SPropDependent = {
               new.formula <-
                   spropMakeFormula(cbGetCurrValue(data, "SPropPFFormula"),
                                    cbGetCurrValue(data, "SPropDependent"))
               data <- cbSetCurrValue(data, "SPropPFFormula", new.formula)
           }
           ,
           SPropIndependent = {
               new.formula <-
                   spropMakeFormula(cbGetCurrValue(data, "SPropPFFormula"),  ,
                                    cbGetCurrValue(data, "SPropIndependent"))
               data <- cbSetCurrValue(data, "SPropPFFormula", new.formula)
           }
           ,
           SPropPFFormula = {
               # Clear dependent and independent current values
               data <- cbSetCurrValue(data, "SPropDependent", "")
               data <- cbSetCurrValue(data, "SPropIndependent", "")
           }
           ,
           SPropPFButton =
           if(exists(cbGetCurrValue(data, "SPropDataFrameList"))) {
               guiModify(classname = "Property",
                         Name = "SPropCFFormulaID",
                         DialogControl = "Invisible",
                         DefaultValue = cbGetDialogId(data))
               guiModify(classname = "Property",
                         Name = "SPropCFData",
                         DialogControl = "Invisible",
                         DefaultValue = cbGetCurrValue(data,
                         "SPropDataFrameList"))
               guiModify(classname = "Property",
                         Name = "SPropCFFormula",
                         DefaultValue = cbGetCurrValue(data, "SPropPFFormula"))
               guiDisplayDialog(classname = "Function", Name = "makeFormulaLm")
               data <- cbSetEnableFlag(data, "SPropDataFrameList", F)
               data <- cbSetEnableFlag(data, "SPropPFFormula", F)
               data <- cbSetEnableFlag(data, "SPropPFButton", F)
               data <- cbSetCurrValue(data, "SPropDependent", "")
               data <- cbSetCurrValue(data, "SPropIndependent", "")
               data <- cbSetEnableFlag(data, "SPropDependent", F)
               data <- cbSetEnableFlag(data, "SPropIndependent", F)
           } else guiCreate("MessageBox", String =
                            "Data Frame doesn't exist. Enter Data Frame before building formula."),
           SPropPFEnableButton = {
               data <- cbSetEnableFlag(data, "SPropDataFrameList", T)
               data <- cbSetEnableFlag(data, "SPropPFFormula", T)
               data <- cbSetEnableFlag(data, "SPropPFButton", T)
               data <- cbSetEnableFlag(data, "SPropDependent", T)
               data <- cbSetEnableFlag(data, "SPropIndependent", T)
           },                           # ignore all the rest
           )
    data
}
