# file MASS/stepAIC.q
# copyright (C) 1994-2002 W. N. Venables and B. D. Ripley
#
stepAIC <-
  function(object, scope, scale = 0,
           direction = c("both", "backward", "forward"),
           trace = 1, keep = NULL, steps = 1000, use.start = F, k = 2, ...)
{
    mydeviance <- function(x, ...)
    {
        dev <- deviance(x)
        if(!is.null(dev)) dev else extractAIC(x, k=0)[2]
    }

    cut.string <- function(string)
    {
        if(length(string) > 1)
            string[-1] <- paste("\n", string[-1], sep = "")
        string
    }

    re.arrange <- function(keep)
    {
        namr <- names(k1 <- keep[[1]])
        namc <- names(keep)
        nc <- length(keep)
        nr <- length(k1)
        array(unlist(keep, recursive = F), c(nr, nc), list(namr, namc))
    }

    step.results <- function(models, fit, object, usingCp=FALSE, keep)
    {
        change <- sapply(models, "[[", "change")
        rd <- sapply(models, "[[", "deviance")
        dd <- c(NA, abs(diff(rd)))
        rdf <- sapply(models, "[[", "df.resid")
        ddf <- c(NA, abs(diff(rdf)))
        AIC <- sapply(models, "[[", "AIC")
        heading <- c("Stepwise Model Path \nAnalysis of Deviance Table",
                     "\nInitial Model:", deparse(as.vector(formula(object))),
                     "\nFinal Model:", deparse(as.vector(formula(fit))),
                     "\n")
        aod <-
            if(usingCp)
                data.frame(Step = change, Df = ddf, Deviance = dd,
                           "Resid. Df" = rdf, "Resid. Dev" = rd,
                           Cp = AIC, check.names = F)
            else data.frame(Step = change, Df = ddf, Deviance = dd,
                            "Resid. Df" = rdf, "Resid. Dev" = rd,
                            AIC = AIC, check.names = F)
        attr(aod, "heading") <- heading
        oldClass(aod) <- c("anova", "data.frame")
        if(is.null(oldClass(fit))) {
            fit <- list(fit = fit, anova = aod)
            if(!is.null(keep)) fit$keep <- keep
        } else {
            fit$anova <- aod
            if(!is.null(keep)) fit$keep <- keep
        }
        fit
    }

    cl <- class(object)[1]
    LM <- (cl == "lm" || cl == "aov" ||
           (cl == "glm" && object$family["name"] == "Gaussian" &&
            object$family["link"] == "Identity: mu"))
    if(!LM && use.start)
        if(is.null(object$linear.predictors)) {
            use.start <- F
            warning("cannot use start with object of class ",
                    oldClass(object))
        } else {
            assign(".eta", object$linear.predictors, frame=1)
            object$call$start <- .eta
    }
    if(cl == "loglm") object$formula <- formula(object$terms)
    md <- missing(direction)
    direction <- match.arg(direction)
    backward <- direction == "both" | direction == "backward"
    forward <- direction == "both" | direction == "forward"
    if(missing(scope)) {
        fdrop <- numeric(0)
        fadd <- attr(Terms(object), "factors")
        if(md) forward <- F
    } else {
        if(is.list(scope)) {
            fdrop <- if(!is.null(fdrop <- scope$lower))
                attr(terms(update.formula(object, fdrop)), "factors")
            else numeric(0)
            fadd <- if(!is.null(fadd <- scope$upper))
                attr(terms(update.formula(object, fadd)), "factors")
        } else {
            fadd <- if(!is.null(fadd <- scope))
                attr(terms(update.formula(object, scope)), "factors")
            fdrop <- numeric(0)
        }
    }
    models <- vector("list", steps)
    if(!is.null(keep)) {
        keep.list <- vector("list", steps)
        nv <- 1
    }
    ## watch out for partial matching here.
    if(is.list(object) && (nmm <- match("nobs", names(object), 0)) > 0)
        n <- object[[nmm]]
    else n <- length(residuals(object))
    fit <- object
    bAIC <- extractAIC(fit, scale, k = k, ...)
    edf <- bAIC[1]
    bAIC <- bAIC[2]
    if(is.na(bAIC))
        stop("AIC is not defined for this model, so stepAIC cannot proceed")
    nm <- 1
    Terms <- Terms(fit)
    if(trace)
        cat("Start:  AIC=", format(round(bAIC, 2)), "\n",
            cut.string(deparse(as.vector(formula(fit)))), "\n\n")

    models[[nm]] <- list(deviance = mydeviance(fit), df.resid = n - edf,
                         change = "", AIC = bAIC)
    if(!is.null(keep)) keep.list[[nm]] <- keep(fit, bAIC)
    usingCp <- F
    while(steps > 0) {
        steps <- steps - 1
        AIC <- bAIC
        bfit <- fit
        ffac <- attr(Terms, "factors")
        ## don't drop strata terms
        if(!is.null(sp <- attr(Terms, "specials")) &&
           !is.null(st <- sp$strata)) ffac <- ffac[-st,]
        scope <- factor.scope(ffac, list(add = fadd, drop = fdrop))
        aod <- NULL
        change <- NULL
        if(backward && length(scope$drop)) {
            aod <- dropterm(fit, scope$drop, scale = scale,
                            trace = max(0, trace - 1), k = k, ...)
            rn <- row.names(aod)
            row.names(aod) <- c(rn[1], paste("-", rn[-1], sep=" "))
            ## drop all zero df terms first.
            if(any(aod$Df == 0, na.rm=T)) {
                zdf <- aod$Df == 0 & !is.na(aod$Df)
                nc <- match(c("Cp", "AIC"), names(aod))
                nc <- nc[!is.na(nc)][1]
                ch <- abs(aod[zdf, nc]) > 0.01
                if(any(ch)) {
                    warning("0 df terms are changing AIC")
                    zdf <- zdf[!ch]
                }
                if(length(zdf) > 0)
                    change <- paste((dimnames(aod)[[1]])[zdf])
            }
        }
        if(is.null(change)) {
            if(forward && length(scope$add)) {
                aodf <- addterm(fit, scope$add, scale = scale,
                                trace = max(0, trace - 1), k = k, ...)
                rn <- row.names(aodf)
                row.names(aodf) <- c(rn[1], paste("+", rn[-1], sep=" "))
                if(is.null(aod)) aod <- aodf
                else {
                    oldClass(aodf) <- "data.frame"
                    ## work around bug in as.anova
                    aod <- rbind(aod, aodf[-1, , drop=F])
                }
            }
            attr(aod, "heading") <- NULL
            if(is.null(aod) || ncol(aod) == 0) break
            ## need to remove any terms with zero df from consideration
            nzdf <- if(!is.null(aod$Df)) aod$Df != 0 | is.na(aod$Df)
            oldClass(aod) <- "data.frame" # work around bug in as.anova
            aod <- aod[nzdf, ]
            if(is.null(aod) || ncol(aod) == 0) break
            nc <- match(c("Cp", "AIC"), names(aod))
            nc <- nc[!is.na(nc)][1]
            o <- order(aod[, nc])
            if(trace) print(aod[o,  ])
            if(o[1] == 1) break
            change <- dimnames(aod)[[1]][o[1]]
        }
        usingCp <- match("Cp", names(aod), 0) > 0
        fit <- update(fit, paste("~ .", change))
        if(is.list(fit) && (nmm <- match("nobs", names(fit), 0)) > 0)
            nnew <- fit[[nmm]]
        else nnew <- length(residuals(fit))
        if(nnew != n)
            stop("number of rows in use has changed: remove missing values?")
        Terms <- Terms(fit)
        bAIC <- extractAIC(fit, scale, k = k, ...)
        edf <- bAIC[1]
        bAIC <- bAIC[2]
        if(trace)
            cat("\nStep:  AIC=", format(round(bAIC, 2)), "\n",
                cut.string(deparse(as.vector(formula(Terms)))), "\n\n")
        if(bAIC >= AIC) break
        nm <- nm + 1
        models[[nm]] <-
            list(deviance = mydeviance(fit), df.resid = n - edf,
                 change = change, AIC = bAIC)
        if(!is.null(keep)) keep.list[[nm]] <- keep(fit, bAIC)
        if(use.start) assign(".eta", fit$linear.predictors, f=1)
    }
    if(!is.null(keep)) keep <- re.arrange(keep.list[seq(nm)])
    if(!LM && use.start) fit$call$start <- NULL
    step.results(models = models[seq(nm)], fit, object, usingCp, keep)
}

extractAIC <- function(fit, scale, k = 2, ...) UseMethod("extractAIC")

extractAIC.coxph <- function(fit, scale, k = 2, ...)
{
    edf <- length(fit$coef)
    if(edf > 0)
        c(edf, -2 * fit$loglik[2] + k * edf)
    else
        c(0, -2 * fit$loglik)
}

extractAIC.survReg <- extractAIC.survreg <-
    function(fit, scale, k = 2, ...)
{
    n <- length(fit$residuals)
    edf <- n  - fit$df.residual
    c(edf, -2 * fit$loglik[2] + k * edf)
}

extractAIC.glm <- function(fit, scale = 0, k = 2, ...)
{
    n <- length(fit$residuals)
    edf <- n  - fit$df.residual
    dev <- fit$deviance
    if(scale > 0) dev <- dev/scale
    if(scale == 0 && fit$family["name"] == "Gaussian") dev <- n * log(dev/n)
    c(edf,  dev + k * edf)
}

extractAIC.aov <-
extractAIC.lm <- function(fit, scale = 0, k = 2, ...)
{
    n <- length(fit$residuals)
    edf <- n  - fit$df.residual
    RSS <- deviance.lm(fit)
    dev <- if(scale > 0) RSS/scale - n else n * log(RSS/n)
    c(edf, dev + k * edf)
}

extractAIC.negbin <- function(fit, scale, k = 2, ...)
{
    n <- length(fit$residuals)
    edf <- n - fit$df.residual
    c(edf,  -fit$twologlik + k * edf)
}

extractAIC.loglm <- function(fit, scale, k = 2, ...)
{
    edf <- fit$n - fit$df
    c(edf,  fit$deviance + k * edf)
}

extractAIC.lme <- function(fit, scale, k = 2, ...)
{
    if(fit$method != "ML") stop("AIC undefined for REML fit")
    res <- logLik(fit)
    edf <- attr(res, "df")
    c(edf,  -2*res + k * edf)
}

extractAIC.gls <- function(fit, scale, k = 2, ...)
{
    res <- logLik(fit)
    edf <- attr(res, "df")
    c(edf,  -2*res + k * edf)
}

Terms <- function(object, ...) UseMethod("Terms")
Terms.default <- function(object, ...) terms(object, ...)

update.lme <-
    function(object, formula, ..., evaluate = T, class)
{
    thisCall <-
        if(missing(formula))
            as.list(match.call(lme, substitute(lme(...))))[-1]
        else
            as.list(match.call(lme, substitute(lme(fixed = formula, ...))))[-1]
    if(is.null(nextCall <- object$origCall) || !is.null(thisCall$fixed) ||
       is.null(thisCall$random)) {
        nextCall <- object$call
    }
    nextCall <- as.list(nextCall)[-1]
    if(is.null(thisCall$random) && is.null(thisCall$subset)) {
        ## no changes in ranef model and no subsetting
        thisCall$random <- object$modelStruct$reStruct
    }
    if(is.na(match("correlation", names(thisCall))) &&
       !is.null(thCor <- object$modelStruct$corStruct)) {
        if(!is.null(thisCall$data) || !is.null(thisCall$subset)
           || !is.null(thisCall$na.action)) {
            ## forcing initialization
            isInitialized(thCor) <- FALSE
        }
        thisCall$correlation <- thCor
    }
    if(is.na(match("weights", names(thisCall))) &&
       !is.null(thWgt <- object$modelStruct$varStruct)) {
        if(!is.null(thisCall$data) || !is.null(thisCall$subset) || !
           is.null(thisCall$na.action)) {
            ## forcing initialization
            isInitialized(thWgt) <- FALSE
        }
        thisCall$weights <- thWgt
    }
    if(!is.null(thisCall$fixed))
        thisCall$fixed <- update(as.formula(nextCall$fixed), formula)
    nextCall[names(thisCall)] <- thisCall
    do.call("lme", nextCall)
}

update.gls <- function(object, formula, ..., evaluate, class)
{
    thisCall <-
        if(missing(formula))
            as.list(match.call(gls, substitute(gls(...))))[-1]
        else
            as.list(match.call(gls, substitute(gls(model = formula, ...))))[-1]
    nextCall <- as.list(object$call)[-1]
    if(is.na(match("correlation", names(thisCall))) &&
       !is.null(thCor <- object$modelStruct$corStruct)) {
        if(!is.null(thisCall$data) || !is.null(thisCall$subset) || !
           is.null(thisCall$na.action)) {
            ## forcing initialization
            isInitialized(thCor) <- FALSE
        }
        thisCall$correlation <- thCor
    }
    if(is.na(match("weights", names(thisCall))) &&
       !is.null(thWgt <- object$modelStruct$varStruct)) {
        if(!is.null(thisCall$data) || !is.null(thisCall$subset)
           || !is.null(thisCall$na.action)) {
            ## forcing initialization
            isInitialized(thWgt) <- FALSE
        }
        thisCall$weights <- thWgt
    }
    if(!is.null(thisCall$model))
        thisCall$model <- update(as.formula(nextCall$model), formula)
    nextCall[names(thisCall)] <- thisCall
    do.call("gls", nextCall)
}
