optim <- function(par, fn, gr = NULL,
                  method = c("Nelder-Mead", "BFGS", "CG", "L-BFGS-B", "SANN"),
                  lower = -Inf, upper = Inf,
                  control = list(), hessian = FALSE, ...)
{
    dots <- list(...)
    fn2 <- Quote(fn(..par))
    if(length(dots)) {
        fn2[names(dots)] <- dots
        fn2 <- as.call(fn2)
    }
    gr2 <- NULL
    if (!is.null(gr)) {
        gr2 <- Quote(gr(..par))
        if(length(dots)) {
            gr2[names(dots)] <- dots
            gr2 <- as.call(gr2)
        }
    }
    method <- match.arg(method)
    if((length(lower) > 1 || length(upper) > 1 ||
       lower[1] != -Inf || upper[1] != Inf)
       && method != "L-BFGS-B") {
        warning("bounds can only be used with method L-BFGS-B")
        method <- "L-BFGS-B"
    }
    ## Defaults :
    con <- list(trace = 0, fnscale = 1, parscale = rep(1, length(par)),
                ndeps = rep(1e-3, length(par)),
                maxit = 100, abstol = -Inf, reltol=sqrt(.Machine$double.eps),
                alpha = 1.0, beta = 0.5, gamma = 2.0,
                REPORT = 10,
                type = 1,
                lmm = 5, factr = 1e7, pgtol = 0,
                tmax = 10, temp = 10.0)
    if (method == "Nelder-Mead") con$maxit <- 500
    if (method == "SANN") con$maxit <- 10000

    con[(namc <- names(control))] <- control
    if (method == "L-BFGS-B" &&
        any(!is.na(match(c("reltol","abstol"), namc))))
        warning("Method L-BFGS-B uses `factr' (& `pgtol') instead of `reltol' and `abstol'")
    npar <- length(par)
    lower <- as.double(rep(lower, , npar))
    upper <- as.double(rep(upper, , npar))
    nm <- names(par)
    res <- .Call("optim", par, fn2, gr2,
                 method, con, lower, upper,  sys.nframe())
    names(res) <- c("par", "value", "counts", "convergence", "message")
    if(!is.null(nm)) names(res$par) <- nm
    names(res$counts) <- c("function", "gradient")
    if (hessian) {
        hess <- .Call("optimhess", res$par, fn2, gr2, con, sys.nframe())
        dim(hess) <- c(npar, npar)
        hess <- 0.5*(hess + t(hess))
        if(!is.null(nm)) dimnames(hess) <- list(nm, nm)
        res$hessian <- hess
    }
    res
}
