arima <- function(x, order = c(0, 0, 0),
                  seasonal = list(order = c(0, 0, 0), period = NA),
                  xreg = NULL, include.mean = T, delta = -1,
                  transform.pars = T, fixed = NULL, init = NULL,
                  method = c("ML", "CSS"), n.cond,
                  optim.control = list())
{
    arma0f <- function(p, fixed, mask)
    {
        par <- as.double(fixed)
        par[mask] <- p
        .Call("arma0fa", par, PACKAGE = "ts")
    }

    arCheck <- function(ar)
    {
        p <- max(which(c(1, -ar) != 0)) - 1
        if(!p) return(T)
        all(Mod(polyroot(c(1, -ar[1:p]))) > 1)
    }

    maInvert <- function(ma)
    {
        q <- length(ma)
        q0 <- max(which(c(1,ma) != 0)) - 1
        if(!q0) return(ma)
        roots <- polyroot(c(1, ma[1:q0]))
        ind <- Mod(roots) < 1
        if(all(!ind)) return(ma)
        if(q0 == 1) return(c(1/ma[1], rep(0, q-q0)))
        roots[ind] <- 1/roots[ind]
        x <- 1
        for(r in roots) x <- c(x, 0) - c(0, x)/r
        c(Re(x[-1]), rep(0, q-q0))
    }

    series <- deparse(substitute(x))
    if(is.matrix(x))
        stop("only implemented for univariate time series")
    method <- match.arg(method)
    x <- as.rts(x)
    dim(x) <- NULL
    n <- length(x)

    if(!missing(order))
        if(!is.numeric(order) || length(order) != 3 || any(order < 0))
            stop("`order' must be a non-negative numeric vector of length 3")
    if(!missing(seasonal))
        if(is.list(seasonal)) {
            if(is.null(seasonal$order))
                stop("`seasonal' must be a list with component `order'")
            if(!is.numeric(seasonal$order) || length(seasonal$order) != 3
               || any(seasonal$order < 0))
                stop("`seasonal$order' must be a non-negative numeric vector of length 3")
        } else if(is.numeric(order)) {
            if(length(order) == 3) seasonal <- list(order=seasonal)
            else ("`seasonal' is of the wrong length")
        } else stop("`seasonal' must be a list with component `order'")

    if(is.null(seasonal$period) || is.na(seasonal$period)
       || seasonal$period == 0) seasonal$period <- frequency(x)
    arma <- c(order[-2], seasonal$order[-2], seasonal$period,
              order[2], seasonal$order[2])
    narma <- sum(arma[1:4])
    if(d <- order[2]) x <- diff(x, 1, d)
    if(d <- seasonal$order[2]) x <- diff(x, seasonal$period, d)
    xtsp <- tspar(x)
    x <- oldUnclass(x)
    nd <- order[2] + seasonal$order[2]
    n.used <- length(x)
    ncond <- n - n.used
    if(method == "CSS") {
        ncond1 <- order[1] + seasonal$period * seasonal$order[1]
        ncond <- if(!missing(n.cond)) ncond + max(n.cond, ncond1)
        else ncond + ncond1
    }
    if(is.null(xreg)) {
        ncxreg <- 0
    } else {
        nmxreg <- deparse(substitute(xreg))
        xreg <- as.matrix(xreg)
        if(nrow(xreg) != n) stop("lengths of x and xreg do not match")
        ncxreg <- ncol(xreg)
    }
    oldClass(xreg) <- NULL
    if (ncxreg > 0 && is.null(dimnames(xreg)[[2]]))
       dimnames(xreg) <- list(NULL,
            if(ncxreg == 1) nmxreg else paste(nmxreg, 1:ncxreg, sep = ""))
    if(include.mean && (nd == 0)) {
        xreg <- cbind(intercept = rep(1, n), xreg = xreg)
        ncxreg <- ncxreg + 1
    }

    if (is.null(fixed)) fixed <- rep(NA, narma + ncxreg)
    else if(length(fixed) != narma + ncxreg) stop("wrong length for fixed")
    mask <- is.na(fixed)
    if(!any(mask)) stop("all parameters were fixed")
    if(transform.pars && any(!mask[1:narma])) {
        warning("some ARMA parameters were fixed: setting transform.pars = F")
        transform.pars <- F
    }

    if(ncxreg) {
        if(d <- order[2]) xreg <- diff(xreg, 1, d)
        if(d <- seasonal$order[2]) xreg <- diff(xreg, seasonal$period, d)
        if(qr(na.omit(xreg))$rank < ncol(xreg)) stop("xreg is collinear")
        cn <- dimnames(xreg)[[2]]
    }
    if(any(is.na(x)) || (ncxreg && any(is.na(xreg)))) {
        ## only exact recursions handle NAs
        delta <- -1
    }

    init0 <- rep(0, narma)
    parscale <- rep(1, narma)
    if (ncxreg) {
        orig.xreg <- (ncxreg == 1) || any(!mask[narma + 1:ncxreg])
        if(!orig.xreg) {
            S <- svd(na.omit(xreg))
            xreg <- xreg %*% S$v
        }
        fit <- lm(x ~ xreg - 1, na.action = na.omit)
        init0 <- c(init0, coef(fit))
        ses <- summary(fit)$coef[,2]
        parscale <- c(parscale, ses)
    }

    storage.mode(x) <- storage.mode(xreg) <- "double"
    if(method == "CSS") transform.pars <- 0
    .Call("setup_starma", as.integer(arma), x, n.used, xreg,
          ncxreg, delta, transform.pars > 0, ncond - (n - n.used))
    on.exit(.Call("free_starma"))

    if(!is.null(init)) {
        if(length(init) != length(init0))
            stop("`init' is of the wrong length")
        if(any(ind <- is.na(init))) init[ind] <- init0[ind]
        if(transform.pars) {
            if(any(!mask[1:narma]))
                warning("transformed ARMA parameters were fixed")
            ## check stationarity
            if(arma[1] > 0)
                if(!arCheck(init[1:arma[1]]))
                    stop("non-stationary AR part")
            if(arma[3] > 0)
                if(!arCheck(init[sum(arma[1:2]) + 1:arma[3]]))
                    stop("non-stationary seasonal AR part")
            ## enforce invertibility
            if(arma[2] > 0) {
                ind <- arma[1] + 1:arma[2]
                init[ind] <- maInvert(init[ind])
            }
            if(arma[4] > 0) {
                ind <- sum(arma[1:3]) + 1:arma[4]
                init[ind] <- maInvert(init[ind])
            }
            init <- .Call("Invtrans", as.double(init))
        }
    } else init <- init0

    .Call("Starma_method", method == "CSS")
    if(!is.element("parscale", names(optim.control)))
        optim.control$parscale <- as.vector(parscale[mask])
    res <- optim(init[mask], arma0f, method = "BFGS",
                 hessian = T, control = optim.control,
                 fixed = fixed, mask = mask)
    if((code <- res$convergence) > 0)
        warning(paste("possible convergence problem: optim gave code=",
                      code))
    coef <- res$par

    if(transform.pars) {
        cf <- fixed
        cf[mask] <- coef
        ## enforce invertibility
        if(arma[2] > 0) {
            ind <- arma[1] + 1:arma[2]
            if(all(mask[ind]))
                cf[ind] <- maInvert(cf[ind])
        }
        if(arma[4] > 0) {
            ind <- sum(arma[1:3]) + 1:arma[4]
            if(all(mask[ind]))
                cf[ind] <- maInvert(cf[ind])
        }
        if(any(cf[mask] != res$par))  {  # need to re-fit
            res <- optim(cf[mask], arma0f, method = "BFGS",
                         hessian = T,
                         control = list(maxit = 0,
                         parscale = optim.control$parscale),
                         fixed = fixed, mask = mask)
            cf[mask] <- res$par
        }
        ## do it this way to ensure hessian was computed inside
        ## stationarity region
        A <- .Call("Gradtrans", as.double(cf))
        dim(A) <- rep(length(fixed), 2)
        A <- A[mask, mask]
        var <- t(A) %*% solve(res$hessian*length(x)) %*% A
        coef <- .Call("Dotrans", as.double(cf))[mask]
        .Call("set_trans", 0)
    } else var <- solve(res$hessian*length(x))
    arma0f(coef, fixed, mask)  # reset pars
    sigma2 <- .Call("get_s2")
    resid <- .Call("get_resid")
    set.ts.class(resid, "rts", xtsp)
    n.used <- sum(!is.na(resid))
    nm <- NULL
    if(arma[1] > 0) nm <- c(nm, paste("ar", 1:arma[1], sep = ""))
    if(arma[2] > 0) nm <- c(nm, paste("ma", 1:arma[2], sep = ""))
    if(arma[3] > 0) nm <- c(nm, paste("sar", 1:arma[3], sep = ""))
    if(arma[4] > 0) nm <- c(nm, paste("sma", 1:arma[4], sep = ""))
    fixed[mask] <- coef
    if(ncxreg > 0) {
        nm <- c(nm, cn)
        if(!orig.xreg) {
            ind <- narma + 1:ncxreg
            fixed[ind] <- S$v %*% fixed[ind]
            A <- diag(narma + ncxreg)
            A[ind, ind] <- S$v
            A <- A[mask, mask]
            var <- A %*% var %*% t(A)
        }
    }
    names(fixed) <- nm
    names(arma) <- c("ar", "ma", "sar", "sma", "period", "diff", "sdiff")
    dimnames(var) <- list(nm[mask], nm[mask])
    value <- 2 * n.used * res$value + n.used + n.used*log(2*pi)
    aic <- if(method != "CSS") value + 2*length(coef) + 2 else NA
    res <- list(coef = fixed, sigma2 = sigma2, var.coef = var, mask = mask,
                loglik = -0.5*value, aic = aic, arma = arma,
                residuals = resid,
                call = match.call(), series = series,
                code = code, n.cond = ncond)
    oldClass(res) <- "Arima"
    res
}

print.Arima <- function(x, digits = max(3, options("digits")$digits - 3),
                         se = T, ...)
{
    cat("\nCall:\n", deparse(x$call), "\n\n", sep = "")
    cat("Coefficients:\n")
    coef <- round(x$coef, digits = digits)
    if (se && nrow(x$var.coef)) {
        ses <- rep(0, length(coef))
        ses[x$mask] <- round(sqrt(diag(x$var.coef)), digits = digits)
        coef <- matrix(coef, 1, dimnames = list(NULL, names(coef)))
        coef <- rbind(coef, s.e. = ses)
    }
    print.default(coef, print.gap = 2)
    cm <- x$call$method
    if(is.null(cm) || cm != "CSS")
        cat("\nsigma^2 estimated as ",
            format(x$sigma2, digits = digits),
            ":  log likelihood = ", format(round(x$loglik,2)),
            ",  aic = ", format(round(x$aic,2)),
            "\n", sep="")
    else
        cat("\nsigma^2 estimated as ",
            format(x$sigma2, digits = digits),
            ":  part log likelihood = ", format(round(x$loglik,2)),
            "\n", sep="")
    invisible(x)
}

predict.Arima <-
    function(object, n.ahead = 1, newxreg = NULL, se.fit=T, ...)
{
    myNCOL <- function(x) if(is.null(x)) 0 else if(is.matrix(x)) ncol(x) else 1
    data <- eval(parse(text = object$series), sys.parent())
    xr <- object$call$xreg
    xreg <- if(!is.null(xr)) eval(xr, sys.parent()) else NULL
    ncxreg <- myNCOL(xreg)
    if(myNCOL(newxreg) != ncxreg)
        stop("xreg and newxreg have different numbers of columns")
    oldClass(xreg) <- NULL
    xtsp <- as.vector(tspar(data)) # drop names
    n <- length(data)
    arma <- object$arma
    coefs <- object$coef
    narma <- sum(arma[1:4])
    if(length(coefs) > narma) {
        if(names(coefs)[narma+1] == "intercept") {
            xreg <- cbind(intercept = rep(1, n), xreg)
            newxreg <- cbind(intercept = rep(1, n.ahead), newxreg)
            ncxreg <- ncxreg+1
        }
        data <- data - as.matrix(xreg) %*% coefs[-(1:narma)]
        xm <- drop(as.matrix(newxreg) %*% coefs[-(1:narma)])
    } else xm <- 0
    ## check invertibility of MA part(s)
    if(arma[2] > 0) {
        ma <- coefs[arma[1] + 1:arma[2]]
        if(any(Mod(polyroot(c(1, ma))) < 1))
            warning("ma part of model is not invertible")
    }
    if(arma[4] > 0) {
        ma <- coefs[sum(arma[1:3]) + 1:arma[4]]
        if(any(Mod(polyroot(c(1, ma))) < 1))
            warning("seasonal ma part of model is not invertible")
    }
    .Call("setup_starma", as.integer(arma), data, n, rep(0, n),
          0, -1, 0, 0)
    on.exit(.Call("free_starma"))
    .Call("Starma_method", T)
    .Call("arma0fa", as.double(coefs))
    z <- .Call("arma0_kfore", arma[6], arma[7], n.ahead)
    pred <- rts(z[[1]] + xm, start = xtsp[1] + n*xtsp[2],
                frequency = xtsp[3])
    if(se.fit) {
        se <- rts(sqrt(z[[2]]),
                 start = xtsp[1] + n*xtsp[2], frequency = xtsp[3])
        return(pred, se)
    } else return(pred)
}

tsdiag <- function(object, gof.lag, ...) UseMethod("tsdiag")

tsdiag.Arima <- function(object, gof.lag = 10, ...)
{
    ## plot standardized residuals, acf of residuals, Ljung-Box p-values
    oldpar<- par(mfrow = c(3, 1))
    on.exit(par(oldpar))
    rs <- object$resid
    stdres <- rs/sqrt(object$sigma2)
    plot(stdres, type = "h", main = "Standardized Residuals", ylab = "")
    abline(h = 0)
    ACF <- acf(rs, plot = F)
    acf.plot(ACF, main = "ACF of Residuals")
    obs <- acf(rs, plot = F, lag.max = gof.lag)$acf
    nlag <- gof.lag
    pval <- numeric(nlag)
    n <- length(rs)
    for(i in 1:nlag) {
        st <- n * (n + 2) * sum(1/seq(n - 1, n - i) * obs[2:(i+1)]^2)
        pval[i] <- 1 - pchisq(st, i)
    }
    plot(1:nlag, pval, xlab = "lag", ylab = "p value", ylim = c(0,1),
         main = "p values for Ljung-Box statistic")
    abline(h = 0.05, lty = 2)
    invisible()
}

ARMAacf <- function(ar = numeric(0), ma = numeric(0), lag.max = r,
                    pacf = F)
{
    p <- length(ar)
    q <- length(ma)
    if(!p && !q) stop("empty model supplied")
    r <- max(p, q + 1)
    if(p > 0) {
        if(r > 1) {
            if(r > p) { ## pad with zeros so p >= q+1
                ar <- c(ar, rep(0, r - p))
                p <- r
            }
            A <- matrix(0, p + 1, 2 * p + 1)
            ind <- as.matrix(expand.grid(1:(p + 1), 1:(p+1)))[, 2:1]
            ind[, 2] <- ind[, 1] + ind[, 2] - 1
            A[ind] <- c(1, -ar)
            A[,  1:p] <- A[, 1:p] + A[, (2 * p + 1):(p + 2)]
            rhs <- c(1, rep(0,p))
            if(q > 0) {
                psi <- c(1, ARMAtoMA(ar, ma, q))
                theta <- c(1, ma, rep(0, q+1))
                for(k in 1 + 0:q) rhs[k] <- sum(psi * theta[k + 0:q])
            }
            ind <- (p+1):1
            Acf <- solve(A[ind, ind], rhs)
	    Acf <- Acf[-1]/Acf[1]
        } else Acf <- ar
        if(lag.max > p) {
            xx <- rep(0, lag.max - p)
            Acf <- c(Acf, filter(xx, rev(ar), "recursive", init = Acf))
        }
        Acf <- c(1, Acf)
    } else if(q > 0) {
        x <- c(1, ma)
        Acf <- filter(c(x, rep(0, q)), rev(x), sides=1)[-(1:q)]
        if(lag.max > q) Acf <- c(Acf, rep(0, lag.max - q))
        Acf <- Acf/Acf[1]
    }
    names(Acf) <- 0:max(p, lag.max)
    if(pacf) .C("uni_pacf", as.double(Acf), pacf = double(lag.max),
                as.integer(lag.max))$pacf
    else Acf
}

ARMAtoMA <- function(ar = numeric(0), ma = numeric(0), lag.max)
    .Call("ARMAtoMA", as.double(ar), as.double(ma), as.integer(lag.max))
