# file MASS/add.q
# copyright (C) 1994-2000 W. N. Venables and B. D. Ripley
#
addterm <-
    function(object, ...) UseMethod("addterm")

addterm.default <-
    function(object, scope, scale = 0, test = c("none", "Chisq"),
             k = 2, sorted = F, trace = F, ...)
{
    if(missing(scope) || is.null(scope)) stop("no terms in scope")
    if(!is.character(scope))
        scope <- add.scope(object, update.formula(object, scope))
    if(!length(scope))
        stop("no terms in scope for adding to object")
#     newform <- update.formula(object,
#                               paste(". ~ . +", paste(scope, collapse="+")))
#     data <- model.frame(update(object, newform)) # remove NAs
#     object <- update(object, data = data)
    ns <- length(scope)
    ans <- matrix(nrow = ns + 1, ncol = 2,
                  dimnames = list(c("<none>", scope), c("df", "AIC")))
    ans[1,  ] <- extractAIC(object, scale, k = k, ...)
    for(i in seq(ns)) {
        tt <- scope[i]
        if(trace) cat("trying +", tt, "\n")
        nfit <- update(object, as.formula(paste("~ . +", tt)))
        ans[i+1,  ] <- extractAIC(nfit, scale, k = k, ...)
    }
    dfs <- ans[,1] - ans[1,1]
    dfs[1] <- NA
    aod <- data.frame(Df = dfs, AIC = ans[,2])
    o <- if(sorted) sort.list(aod$AIC) else seq(along=aod$AIC)
    test <- match.arg(test)
    if(test == "Chisq") {
        dev <- ans[,2] - k*ans[, 1]
        dev <- dev[1] - dev; dev[1] <- NA
        nas <- !is.na(dev)
        P <- dev
        P[nas] <- 1 - pchisq(dev[nas], dfs[nas])
        aod[, c("LRT", "Pr(Chi)")] <- list(dev, P)
    }
    aod <- aod[o, ]
    head <- c("Single term additions", "\nModel:",
              deparse(as.vector(formula(object))))
    if(scale > 0)
        head <- c(head, paste("\nscale: ", format(scale), "\n"))
    oldClass(aod) <- c("anova", "data.frame")
    attr(aod, "heading") <- head
    aod
}

addterm.lm <-
  function(object, scope, scale = 0, test = c("none", "Chisq", "F"),
           k = 2, sorted = F, ...)
{
    Fstat <- function(table, RSS, rdf) {
        dev <- table$"Sum of Sq"
        df <- table$Df
        rms <- (RSS - dev)/(rdf - df)
        Fs <- (dev/df)/rms
        Fs[df < 1e-4] <- NA
        P <- Fs
        nnas <- !is.na(Fs)
        P[nnas] <- 1 - pf(Fs[nnas], df[nnas], rdf - df[nnas])
        list(Fs=Fs, P=P)
    }

    if(missing(scope) || is.null(scope)) stop("no terms in scope")
    aod <- add1.lm(object, scope=scope, scale=scale)[ , -4]
    dfs <- c(0, aod$Df[-1]) + object$rank; RSS <- aod$RSS
    n <- length(object$residuals)
    if(scale > 0) aic <- RSS/scale - n + k*dfs
    else aic <- n * log(RSS/n) + k*dfs
    aod$AIC <- aic
    o <- if(sorted) sort.list(aod$AIC) else seq(along=aod$AIC)
    if(scale > 0) names(aod) <- c("Df", "Sum of Sq", "RSS", "Cp")
    test <- match.arg(test)
    if(test == "Chisq") {
        dev <- aod$"Sum of Sq"
        if(scale == 0) {
            dev <- n * log(RSS/n)
            dev <- dev[1] - dev
            dev[1] <- NA
        } else dev <- dev/scale
        df <- aod$Df
        nas <- !is.na(df)
        dev[nas] <- 1 - pchisq(dev[nas], df[nas])
        aod[, "Pr(Chi)"] <- dev
    } else if(test == "F") {
        rdf <- object$df.resid
        aod[, c("F Value", "Pr(F)")] <- Fstat(aod, aod$RSS[1], rdf)
    }
    aod <- aod[o, ]
    head <- c("Single term additions", "\nModel:",
              deparse(as.vector(formula(object))))
    if(scale > 0)
        head <- c(head, paste("\nscale: ", format(scale), "\n"))
    oldClass(aod) <- c("anova", "data.frame")
    attr(aod, "heading") <- head
    aod
}

addterm.negbin <- addterm.survreg <-
  function(object, ...)  addterm.default(object, ...)

addterm.glm <-
  function(object, scope, scale = 0, test = c("none", "Chisq", "F"),
           k = 2, sorted = F, trace = F, ...)
{
    Fstat <- function(table, rdf) {
	dev <- table$Deviance
	df <- table$Df
	diff <- pmax(0, (dev[1] - dev)/df)
	Fs <- (diff/df)/(dev/(rdf-df))
	Fs[df < .Machine$double.eps] <- NA
	P <- Fs
	nnas <- !is.na(Fs)
	P[nnas] <- 1 - pf(Fs[nnas], df[nnas], rdf - df[nnas])
	list(Fs=Fs, P=P)
    }
    if(missing(scope) || is.null(scope)) stop("no terms in scope")
    if(!is.character(scope))
        scope <- add.scope(object, update.formula(object, scope))
    if(!length(scope))
        stop("no terms in scope for adding to object")
    oTerms <- attr(Terms(object), "term.labels")
    int <- attr(Terms(object), "intercept")
    ns <- length(scope)
    dfs <- dev <- numeric(ns+1)
    names(dfs) <- names(dev) <- c("<none>", scope)
    dfs[1] <- object$rank
    dev[1] <- object$deviance
    add.rhs <- paste(scope, collapse = "+")
    add.rhs <- eval(parse(text = paste("~ . +", add.rhs)))
    new.form <- update.formula(object, add.rhs)
    oc <- object$call
    Terms <- terms(new.form)
    oc$formula <- Terms
    fob <- list(call = oc)
    oldClass(fob) <- oldClass(object)
    x <- model.matrix(Terms, model.frame(fob, xlevels = attr(object, "xlevels")),
                      contrasts = object$contrasts)
    n <- nrow(x)
    m <- model.frame(object)
    oldn <- length(object$residuals)
    y <- model.extract(m, "response")
    newn <- length(y)
    if(newn < oldn)
        warning("using the ", newn, "/", oldn , " rows from a combined fit")
    wt <- object$prior.weights
    if(is.null(wt)) wt <- rep(1, n)
    start <- object$call$start
    control <- object$call$control
    if(!is.null(control)) control <- eval(control)
    else {
        extras <- match.call(glm, object$call, F)$...
        if(length(extras)) {
            extras[[1]]<- as.name("glm.control")
            control <- eval(extras)
        } else control <-  glm.control()
    }
    offset <- model.extract(m, offset)
    asgn <- attr(x, "assign")
    tl <- names(asgn)
    if(!all(match(scope, tl, F)))
        stop("scope is not a subset of term labels of the supplied x")
    oldx <- unlist(asgn[oTerms])
    if(int) oldx <- c(int, oldx)
    for(tt in scope) {
        if(trace) cat("trying +", tt, "\n")
        X <- x[, c(oldx, asgn[[tt]]), drop = F]
        z <-  glm.fit(X, y, wt, start = start, offset = offset,
                      family = family(object), maxit = control$maxit,
                      epsilon = control$epsilon, trace = max(0, trace-1))
        dfs[tt] <- z$rank
        dev[tt] <- z$deviance
    }
    if (is.null(scale) || scale == 0)
        dispersion <- summary(object, dispersion = NULL)$dispersion
    else dispersion <- scale
    if(object$family["name"] == "Gaussian") {
        if(scale > 0) loglik <- dev/scale - n
        else loglik <- n * log(dev/n)
    } else loglik <- dev/dispersion
    aic <- loglik + k * dfs
    dfs <- dfs - dfs[1]
    dfs[1] <- NA
    aod <- data.frame(Df = dfs, Deviance = dev, AIC = aic,
                      row.names = names(dfs))
    o <- if(sorted) sort.list(aod$AIC) else seq(along=aod$AIC)
    if(all(is.na(aic))) aod <- aod[, -3]
    test <- match.arg(test)
    if(test == "Chisq") {
        dev <- pmax(0, loglik[1] - loglik)
        dev[1] <- NA
        LRT <- if(dispersion == 1) "LRT" else "scaled dev."
        aod[, LRT] <- dev
        nas <- !is.na(dev)
        dev[nas] <- 1 - pchisq(dev[nas], aod$Df[nas])
        aod[, "Pr(Chi)"] <- dev
    } else if(test == "F") {
	rdf <- object$df.residual
	aod[, c("F value", "Pr(F)")] <- Fstat(aod, rdf)
    }
    aod <- aod[o, ]
    head <- c("Single term additions", "\nModel:",
              deparse(as.vector(formula(object))))
    if(scale > 0)
        head <- c(head, paste("\nscale: ", format(scale), "\n"))
    oldClass(aod) <- c("anova", "data.frame")
    attr(aod, "heading") <- head
    aod
}

addterm.mlm <- function(object, ...)
    stop("no addterm method implemented for mlm models")

dropterm <- function(object, ...) UseMethod("dropterm")

dropterm.default <-
  function(object, scope, scale = 0, test = c("none", "Chisq"),
           k = 2, sorted = F, trace = F, ...)
{
    tl <- attr(Terms(object), "term.labels")
    if(missing(scope)) scope <- drop.scope(object)
    else {
        if(!is.character(scope))
            scope <- attr(terms(update.formula(object, scope)), "term.labels")
        if(!all(match(scope, tl, F)))
            stop("scope is not a subset of term labels")
    }
#    data <- model.frame(object) # remove NAs
#    object <- update(object, data = data)
    ns <- length(scope)
    ans <- matrix(nrow = ns + 1, ncol = 2,
                  dimnames =  list(c("<none>", scope), c("df", "AIC")))
    ans[1,  ] <- extractAIC(object, scale, k = k, ...)
    for(i in seq(ns)) {
        tt <- scope[i]
        if(trace) cat("trying -", tt, "\n")
        nfit <- update(object, as.formula(paste("~ . -", tt)))
        ans[i+1,  ] <- extractAIC(nfit, scale, k = k, ...)
    }
    dfs <- ans[1,1] - ans[,1]
    dfs[1] <- NA
    aod <- data.frame(Df = dfs, AIC = ans[,2])
    o <- if(sorted) sort.list(aod$AIC) else seq(along=aod$AIC)
    test <- match.arg(test)
    if(test == "Chisq") {
        dev <- ans[, 2] - k*ans[, 1]
        dev <- dev - dev[1] ; dev[1] <- NA
        nas <- !is.na(dev)
        P <- dev
        P[nas] <- 1 - pchisq(dev[nas], dfs[nas])
        aod[, c("LRT", "Pr(Chi)")] <- list(dev, P)
    }
    aod <- aod[o, ]
    head <- c("Single term deletions", "\nModel:",
              deparse(as.vector(formula(object))))
    if(scale > 0)
        head <- c(head, paste("\nscale: ", format(scale), "\n"))
    oldClass(aod) <- c("anova", "data.frame")
    attr(aod, "heading") <- head
    aod
}

dropterm.lm <-
  function(object, scope = drop.scope(object), scale = 0,
           test = c("none", "Chisq", "F"),
           k = 2, sorted = F, ...)
{
    aod <- drop1.lm(object, scope=scope, scale=scale)[, -4]
    dfs <-  object$rank - c(0, aod$Df[-1]); RSS <- aod$RSS
    n <- length(object$residuals)
    aod$AIC <- if(scale > 0)RSS/scale - n + k*dfs
    else n * log(RSS/n) + k*dfs
    o <- if(sorted) sort.list(aod$AIC) else seq(along=aod$AIC)
    if(scale > 0) names(aod) <- c("Df", "Sum of Sq", "RSS", "Cp")
    test <- match.arg(test)
    if(test == "Chisq") {
        dev <- aod$"Sum of Sq"
        nas <- !is.na(dev)
        dev[nas] <- 1 - pchisq(dev[nas]/scale, aod$Df[nas])
        aod[, "Pr(Chi)"] <- dev
    } else if(test == "F") {
        rdf <- object$df.resid
        aod <- stat.anova(aod, test = "F", scale = aod$RSS[1]/rdf, df.scale = rdf)
    }
    aod <- aod[o, ]
    head <- c("Single term deletions", "\nModel:",
              deparse(as.vector(formula(object))))
    if(scale > 0)
        head <- c(head, paste("\nscale: ", format(scale), "\n"))
    oldClass(aod) <- c("anova", "data.frame")
    attr(aod, "heading") <- head
    aod
}

dropterm.mlm <- function(object, ...)
  stop("dropterm not implemented for mlm models")

dropterm.glm <-
  function(object, scope, scale = 0, test = c("none", "Chisq", "F"),
           k = 2, sorted = F, trace = F, ...)
{
    x <- model.matrix(object)
    iswt <- !is.null(wt <- object$weights)
    n <- nrow(x)
    tl <- attr(object$terms, "term.labels")
    if(missing(scope)) scope <- drop.scope(object)
    else {
        if(!is.character(scope))
            scope <- attr(terms(update.formula(object, scope)), "term.labels")
        if(!all(match(scope, tl, F)))
            stop("scope is not a subset of term labels")
  }
    ns <- length(scope)
    asgn <- attr(x, "assign")[scope]
    rdf <- object$df.resid
    chisq <- object$deviance
    dfs <- numeric(ns)
    dev <- numeric(ns)
    y <- object$y
    m <- model.frame(object)
    if(is.null(y)) y <- model.extract(m, "response")
    wt <- object$prior.weights
    if(is.null(wt)) wt <- rep(1, n)
    start <- object$call$start
    rank <- object$rank
    control <- object$call$control
    if(!is.null(control)) control <- eval(control)
    else {
        extras <- match.call(glm, object$call, F)$...
        if(length(extras)) {
            extras[[1]]<- as.name("glm.control")
            control <- eval(extras)
        } else control <-  glm.control()
    }
    offset <- model.extract(m, offset)
    for(i in 1:ns) {
        if(trace) cat("trying -", scope[i], "\n")
        jj <- setdiff(seq(ncol(x)), asgn[[i]])
        z <-  glm.fit(x[, jj, drop = F], y, wt, start = start, offset = offset,
                      family = family(object), maxit = control$maxit,
                      epsilon = control$epsilon, trace = max(0, trace-1))
        dfs[i] <- z$rank
        dev[i] <- z$deviance
    }
    scope <- c("<none>", scope)
    dfs <- c(object$rank, dfs)
    dev <- c(chisq, dev)
    if (is.null(scale) || scale == 0)
        dispersion <- summary(object, dispersion = NULL)$dispersion
    else dispersion <- scale
    if(object$family["name"] == "Gaussian") {
        if(scale > 0) loglik <- dev/scale - n
        else loglik <- n * log(dev/n)
    } else loglik <- dev/dispersion
    aic <- loglik + k * dfs
    dfs <- dfs[1] - dfs
    dfs[1] <- NA
    aod <- data.frame(Df = dfs, Deviance = dev, AIC = aic, row.names = scope)
    o <- if(sorted) sort.list(aod$AIC) else seq(along=aod$AIC)
    if(all(is.na(aic))) aod <- aod[, -3]
    test <- match.arg(test)
    if(test == "Chisq") {
        dev <- pmax(0, loglik - loglik[1])
        dev[1] <- NA
        nas <- !is.na(dev)
        LRT <- if(dispersion == 1) "LRT" else "scaled dev."
        aod[, LRT] <- dev
        dev[nas] <- 1 - pchisq(dev[nas], aod$Df[nas])
        aod[, "Pr(Chi)"] <- dev
    } else if(test == "F") {
        fam <- object$family$family  ## extra line needed
        if(fam == "binomial" || fam == "poisson")
            warning(paste("F test assumes quasi", fam, " family", sep=""))
	dev <- aod$Deviance
	rms <- dev[1]/rdf
        dev <- pmax(0, dev - dev[1])
	dfs <- aod$Df
	rdf <- object$df.residual
	Fs <- (dev/dfs)/rms
	Fs[dfs < 1e-4] <- NA
	P <- Fs
	nas <- !is.na(Fs)
	P[nas] <- pf(Fs[nas], dfs[nas], rdf, lower.tail=FALSE)
	aod[, c("F value", "Pr(F)")] <- list(Fs, P)
    }
    aod <- aod[o, ]
    head <- c("Single term deletions", "\nModel:",
              deparse(as.vector(formula(object))))
    if(scale > 0)
        head <- c(head, paste("\nscale: ", format(scale), "\n"))
    oldClass(aod) <- c("anova", "data.frame")
    attr(aod, "heading") <- head
    aod
}

dropterm.negbin <- dropterm.survreg <-
    function(object, ...) dropterm.default(object, ...)
