/* int vs long? */
/* What's a real definition of isNull? */

#include <S.h>
#include <string.h>

/* some platforms define finite in ieeefp.h or elsewhere
int finite(double value);
*/
#include <ieee.h>
#define FINITE _dfinite

#define isNull(x)   (GET_LENGTH(x) == 0)

typedef  s_object * SEXP;

void setulb(int n, int m, double *x, double *l, double *u, int *nbd,
	    double *f, double *g, double factr, double *pgtol,
	    double *wa, int * iwa, char *task, int iprint,
	    int *lsave, int *isave, double *dsave);

static SEXP getListElement(SEXP list, char *str)
{
    SEXP elmt, names = GET_NAMES(list);
    int i;

    for (i = 0; i < GET_LENGTH(list); i++)
	if (strcmp(CHARACTER_DATA(names)[i], str) == 0) {
	    elmt = LIST_POINTER(list)[i];
	    break;
	}
    return elmt;
}

static double * vect(int n)
{
    return Salloc(n, double);
}

typedef double optimfn(int, double *, void *);
typedef void optimgr(int, double *, double *, void *);


typedef struct opt_struct
{
    SEXP R_fcall;    /* function */
    SEXP R_gcall;    /* gradient */
    int frame;      /* where to evaluate the calls */
    double* ndeps;   /* tolerances for numerical derivatives */
    double fnscale;  /* scaling for objective */
    double* parscale;/* scaling for parameters */
    int usebounds;
    double* lower, *upper;
} opt_struct, *OptStruct;

static void vmmin(int n, double *b, double *Fmin,
		  optimfn fn, optimgr gr, int maxit, int trace,
		  int *mask, double abstol, double reltol, int nREPORT,
		  void *ex, int *fncount, int *grcount, int *fail);
static void nmmin(int n, double *Bvec, double *X, double *Fmin, optimfn fn,
		  int *fail, double abstol, double intol, void *ex,
		  double alpha, double beta, double gamm, int trace,
		  int *fncount, int maxit);
static void cgmin(int n, double *Bvec, double *X, double *Fmin,
		  optimfn fn, optimgr gr,
		  int *fail, double abstol, double intol, void *ex,
		  int type, int trace, int *fncount, int *grcount, int maxit);
static void lbfgsb(int n, int m, double *x, double *l, double *u, int *nbd,
		   double *Fmin, optimfn fn, optimgr gr, int *fail, void *ex,
		   double factr, double pgtol, int *fncount, int *grcount,
		   int maxit, char *msg, int trace, int nREPORT);
static void samin(int n, double *pb, double *yb, optimfn fn, int maxit,
		  int tmax, double ti, int trace, void *ex);


static double fminfn(int n, double *p, void *ex)
{
    S_EVALUATOR
    SEXP s, x;
    int i;
    double val;
    OptStruct OS = (OptStruct) ex;

    /* for(i = 0; i < n; i++) printf(" %f", p[i]); printf("\n"); */
    x = NEW_NUMERIC(n);
    for (i = 0; i < n; i++) {
	if (!FINITE(p[i]))
	    PROBLEM "non-finite value supplied by optim" ERROR;
	NUMERIC_POINTER(x)[i] = p[i] * (OS->parscale[i]);
    }

    ASSIGN_IN_FRAME("..par", x, OS->frame);
    s = AS_NUMERIC(EVAL_IN_FRAME(OS->R_fcall, OS->frame));
    val = NUMERIC_POINTER(s)[0]/(OS->fnscale);
    return val;
}

static void fmingr(int n, double *p, double *df, void *ex)
{
    S_EVALUATOR
    SEXP s, x;
    int i;
    double val1, val2, eps, epsused, tmp;
    OptStruct OS = (OptStruct) ex;

    if (!isNull(OS->R_gcall)) { /* analytical derivatives */
	x = NEW_NUMERIC(n);
	for (i = 0; i < n; i++) {
	    if (!FINITE(p[i]))
		PROBLEM "non-finite value supplied to gradient" ERROR;
	    NUMERIC_POINTER(x)[i] = p[i] * (OS->parscale[i]);
	}
	ASSIGN_IN_FRAME("..par", x, OS->frame);
	s = AS_NUMERIC(EVAL_IN_FRAME(OS->R_gcall, OS->frame));
	if(GET_LENGTH(s) != n)
	    PROBLEM
		"gradient in optim evaluated to length %d not %d",
		GET_LENGTH(s), n ERROR;
	for (i = 0; i < n; i++)
	    df[i] = NUMERIC_POINTER(s)[i] * (OS->parscale[i])/(OS->fnscale);
    } else { /* numerical derivatives */
	x = NEW_NUMERIC(n);
	for (i = 0; i < n; i++)
	    NUMERIC_POINTER(x)[i] = p[i] * (OS->parscale[i]);
	if(OS->usebounds == 0) {
	    for (i = 0; i < n; i++) {
		eps = OS->ndeps[i];
		NUMERIC_POINTER(x)[i] = (p[i] + eps) * (OS->parscale[i]);
		ASSIGN_IN_FRAME("..par", x, OS->frame);
		s = EVAL_IN_FRAME(OS->R_fcall, OS->frame);
		s = AS_NUMERIC(s);
		val1 = NUMERIC_POINTER(s)[0]/(OS->fnscale);
		NUMERIC_POINTER(x)[i] = (p[i] - eps) * (OS->parscale[i]);
		ASSIGN_IN_FRAME("..par", x, OS->frame);
		s = EVAL_IN_FRAME(OS->R_fcall, OS->frame);
		s = AS_NUMERIC(s);
		val2 = NUMERIC_POINTER(s)[0]/(OS->fnscale);
		df[i] = (val1 - val2)/(2 * eps);
		if(!FINITE(df[i]))
		    PROBLEM "non-finite finite-difference value [%d]", i
			ERROR;
		NUMERIC_POINTER(x)[i] = p[i] * (OS->parscale[i]);
	    }
	} else { /* usebounds */
	    for (i = 0; i < n; i++) {
		epsused = eps = OS->ndeps[i];
		tmp = p[i] + eps;
		if (tmp > OS->upper[i]) {
		    tmp = OS->upper[i];
		    epsused = tmp - p[i] ;
		}
		NUMERIC_POINTER(x)[i] = tmp * (OS->parscale[i]);
		ASSIGN_IN_FRAME("..par", x, OS->frame);
		s = EVAL_IN_FRAME(OS->R_fcall, OS->frame);
		s = AS_NUMERIC(s);
		val1 = NUMERIC_POINTER(s)[0]/(OS->fnscale);
		tmp = p[i] - eps;
		if (tmp < OS->lower[i]) {
		    tmp = OS->lower[i];
		    eps = p[i] - tmp;
		}
		NUMERIC_POINTER(x)[i] = tmp * (OS->parscale[i]);
		ASSIGN_IN_FRAME("..par", x, OS->frame);
		s = EVAL_IN_FRAME(OS->R_fcall, OS->frame);
		s = AS_NUMERIC(s);
		val2 = NUMERIC_POINTER(s)[0]/(OS->fnscale);
		df[i] = (val1 - val2)/(epsused + eps);
		if(!FINITE(df[i]))
		    PROBLEM "non-finite finite-difference value [%d]", i
			ERROR;
		NUMERIC_POINTER(x)[i] = p[i] * (OS->parscale[i]);
	    }
	}
    }
}

SEXP optim(SEXP par, SEXP fn, SEXP gr, SEXP method, SEXP options,
	   SEXP slower, SEXP supper, SEXP sframe)
{
    S_EVALUATOR
    SEXP tmp, res, value, counts, conv;
    int i, npar=0, *mask, trace, maxit, fncount, grcount, nREPORT;
    int ifail = 0;
    double *dpar, *opar, val, abstol, reltol;
    char *tn;
    OptStruct OS;

    OS = Salloc(1, opt_struct);
    OS->usebounds = 0;
    OS->frame = (int) INTEGER_VALUE(sframe);
    tn = CHARACTER_DATA(method)[0];
    OS->R_fcall = fn;
    par = AS_NUMERIC(COPY(par));
    npar = GET_LENGTH(par);
    dpar = vect(npar);
    opar = vect(npar);
    trace = (int) INTEGER_VALUE(getListElement(options, "trace"));
    OS->fnscale = NUMERIC_VALUE(getListElement(options, "fnscale"));
    tmp = getListElement(options, "parscale");
    tmp = AS_NUMERIC(tmp);
    if (GET_LENGTH(tmp) != npar)
	 PROBLEM "parscale is of the wrong length" ERROR;
    OS->parscale = vect(npar);
    for (i = 0; i < npar; i++) OS->parscale[i] = NUMERIC_POINTER(tmp)[i];
    for (i = 0; i < npar; i++)
	dpar[i] = NUMERIC_POINTER(par)[i] / (OS->parscale[i]);
    res = NEW_LIST(5);
    value = NEW_NUMERIC(1);
    counts = NEW_INTEGER(2);
    conv = NEW_INTEGER(1);
    abstol = NUMERIC_VALUE(getListElement(options, "abstol"));
    reltol = NUMERIC_VALUE(getListElement(options, "reltol"));
    maxit = (int) INTEGER_VALUE(getListElement(options, "maxit"));

    if (strcmp(tn, "Nelder-Mead") == 0) {
	double alpha, beta, gamm;

	alpha = NUMERIC_VALUE(getListElement(options, "alpha"));
	beta = NUMERIC_VALUE(getListElement(options, "beta"));
	gamm = NUMERIC_VALUE(getListElement(options, "gamma"));
	nmmin(npar, dpar, opar, &val, fminfn, &ifail, abstol, reltol,
	      (void *)OS, alpha, beta, gamm, trace, &fncount, maxit);
	for (i = 0; i < npar; i++)
	    NUMERIC_POINTER(par)[i] = opar[i] * (OS->parscale[i]);
	na_set(&grcount, S_MODE_INT);

    } else if (strcmp(tn, "SANN") == 0) {
	int tmax;
	double temp;
	tmax = INTEGER_VALUE(getListElement(options, "tmax"));
	temp = NUMERIC_VALUE(getListElement(options, "temp"));
	if (is_na_INT(&tmax)) PROBLEM "tmax is not an integer" ERROR;
	samin (npar, dpar, &val, fminfn, maxit, tmax, temp, trace,
	       (void *)OS);
	for (i = 0; i < npar; i++)
	    NUMERIC_POINTER(par)[i] = dpar[i] * (OS->parscale[i]);
	fncount = maxit;
	na_set(&grcount, S_MODE_INT);

    } else if (strcmp(tn, "BFGS") == 0) {
	SEXP ndeps;

	nREPORT = (int) INTEGER_VALUE(getListElement(options, "REPORT"));
	OS->R_gcall = gr;
	if (isNull(gr)) {
	    ndeps = getListElement(options, "ndeps");
	    if (GET_LENGTH(ndeps) != npar)
		 PROBLEM "ndeps is of the wrong length" ERROR;
	    OS->ndeps = vect(npar);
	    ndeps = AS_NUMERIC(ndeps);
	    for (i = 0; i < npar; i++)
		OS->ndeps[i] = NUMERIC_POINTER(ndeps)[i];
	}
	mask = Salloc(npar, int);
	for (i = 0; i < npar; i++) mask[i] = 1;
	vmmin(npar, dpar, &val, fminfn, fmingr, maxit, trace, mask, abstol,
	      reltol, nREPORT, (void *)OS, &fncount, &grcount, &ifail);
	for (i = 0; i < npar; i++)
	    NUMERIC_POINTER(par)[i] = dpar[i] * (OS->parscale[i]);
    } else if (strcmp(tn, "CG") == 0) {
	int type;
	SEXP ndeps;

	type = (int) INTEGER_VALUE(getListElement(options, "type"));
	OS->R_gcall = gr;
	if (isNull(gr)) {
	    ndeps = getListElement(options, "ndeps");
	    if (GET_LENGTH(ndeps) != npar)
		PROBLEM "ndeps is of the wrong length" ERROR;
	    OS->ndeps = vect(npar);
	    ndeps = AS_NUMERIC(ndeps);
	    for (i = 0; i < npar; i++)
		OS->ndeps[i] = NUMERIC_POINTER(ndeps)[i];
	}
	cgmin(npar, dpar, opar, &val, fminfn, fmingr, &ifail, abstol,
	      reltol, (void *)OS, type, trace, &fncount, &grcount, maxit);
	for (i = 0; i < npar; i++)
	    NUMERIC_POINTER(par)[i] = opar[i] * (OS->parscale[i]);

    } else if (strcmp(tn, "L-BFGS-B") == 0) {
	SEXP ndeps, smsg;
	double *lower = vect(npar), *upper = vect(npar);
	int lmm, *nbd = Salloc(npar, int);
	double factr, pgtol;
	static char msg[60]; /* passed back up */

	nREPORT = (int) INTEGER_VALUE(getListElement(options, "REPORT"));
	factr = NUMERIC_VALUE(getListElement(options, "factr"));
	pgtol = NUMERIC_VALUE(getListElement(options, "pgtol"));
	lmm = (int) INTEGER_VALUE(getListElement(options, "lmm"));
	OS->R_gcall = gr;
	if (isNull(gr)) {
	    ndeps = getListElement(options, "ndeps");
	    if (GET_LENGTH(ndeps) != npar)
		PROBLEM "ndeps is of the wrong length" ERROR;
	    OS->ndeps = vect(npar);
	    ndeps = AS_NUMERIC(ndeps);
	    for (i = 0; i < npar; i++)
		OS->ndeps[i] = NUMERIC_POINTER(ndeps)[i];
	}
	for (i = 0; i < npar; i++) {
	    lower[i] = NUMERIC_POINTER(slower)[i] / (OS->parscale[i]);
	    upper[i] = NUMERIC_POINTER(supper)[i] / (OS->parscale[i]);
	    if (!FINITE(lower[i])) {
		if (!FINITE(upper[i])) nbd[i] = 0; else nbd[i] = 3;
	    } else {
		if (!FINITE(upper[i])) nbd[i] = 1; else nbd[i] = 2;
	    }
	}
	OS->usebounds = 1;
	OS->lower = lower;
	OS->upper = upper;
	lbfgsb(npar, lmm, dpar, lower, upper, nbd, &val, fminfn, fmingr,
	       &ifail, (void *)OS, factr, pgtol, &fncount, &grcount,
	       maxit, msg, trace, nREPORT);
	for (i = 0; i < npar; i++)
	    NUMERIC_POINTER(par)[i] = dpar[i] * (OS->parscale[i]);
	smsg = NEW_CHARACTER(1);
	CHARACTER_DATA(smsg)[0] = msg;
	SET_ELEMENT(res, 4, smsg);
    } else
	 PROBLEM "unknown method" ERROR;

    NUMERIC_POINTER(value)[0] = val * (OS->fnscale);
    SET_ELEMENT(res, 0, par);
    SET_ELEMENT(res, 1, value);
    INTEGER_POINTER(counts)[0] = (long) fncount;
    INTEGER_POINTER(counts)[1] = (long) grcount;
    SET_ELEMENT(res, 2, counts);
    INTEGER_POINTER(conv)[0] = (long) ifail;
    SET_ELEMENT(res, 3, conv);
    return res;
}

SEXP optimhess(SEXP par, SEXP fn, SEXP gr, SEXP options, SEXP sframe)
{
    S_EVALUATOR
    SEXP tmp, ndeps, ans;
    OptStruct OS;
    int npar, i , j;
    double *dpar, *df1, *df2, eps;

    OS = Salloc(1, opt_struct);
    OS->usebounds = 0;
    OS->frame = (int) INTEGER_VALUE(sframe);
    par = AS_NUMERIC(par); /* strips names */
    npar = GET_LENGTH(par);
    OS->fnscale = NUMERIC_VALUE(getListElement(options, "fnscale"));
    tmp = getListElement(options, "parscale");
    tmp = AS_NUMERIC(tmp);
    if (GET_LENGTH(tmp) != npar)
	 PROBLEM "parscale is of the wrong length" ERROR;
    OS->parscale = vect(npar);
    for (i = 0; i < npar; i++) OS->parscale[i] = NUMERIC_POINTER(tmp)[i];
    OS->R_fcall = fn;
    OS->R_gcall = gr;

    ndeps = getListElement(options, "ndeps");
    if (GET_LENGTH(ndeps) != npar)
	PROBLEM "ndeps is of the wrong length" ERROR;
    OS->ndeps = vect(npar);
    ndeps = AS_NUMERIC(ndeps);
    for (i = 0; i < npar; i++) OS->ndeps[i] = NUMERIC_POINTER(ndeps)[i];
    ans = NEW_NUMERIC(npar * npar);
    dpar = vect(npar);
    for (i = 0; i < npar; i++)
	dpar[i] = NUMERIC_POINTER(par)[i] / (OS->parscale[i]);
    df1 = vect(npar);
    df2 = vect(npar);
    for (i = 0; i < npar; i++) {
	eps = OS->ndeps[i]/(OS->parscale[i]);
	dpar[i] = dpar[i] + eps;
	fmingr(npar, dpar, df1, (void *)OS);
	dpar[i] = dpar[i] - 2 * eps;
	fmingr(npar, dpar, df2, (void *)OS);
	for (j = 0; j < npar; j++)
	    NUMERIC_POINTER(ans)[i * npar + j] =
		(OS->fnscale) * (df1[j] - df2[j])/
		(2 * eps * (OS->parscale[i]) * (OS->parscale[j]));
	dpar[i] = dpar[i] + eps;
    }
    return ans;
}


static double ** matrix(int nrh, int nch)
{
    int   i;
    double **m;

    m = Salloc((nrh + 1), double *);
    for (i = 0; i <= nrh; i++)
	m[i] = Salloc((nch + 1), double);
    return m;
}

static double ** Lmatrix(int n)
{
    int   i;
    double **m;

    m = Salloc(n,double *);
    for (i = 0; i < n; i++)
	m[i] = Salloc((i + 1), double);
    return m;
}



#define stepredn	0.2
#define acctol		0.0001
#define reltest		10.0


/*  BFGS variable-metric method, based on Pascal code
in J.C. Nash, `Compact Numerical Methods for Computers', 2nd edition,
converted by p2c then re-crafted by B.D. Ripley */

static void
vmmin(int n0, double *b, double *Fmin, optimfn fminfn, optimgr fmingr,
      int maxit, int trace, int *mask,
      double abstol, double reltol, int nREPORT, void *ex,
      int *fncount, int *grcount, int *fail)
{
    s_boolean accpoint, enough;
    double *g, *t, *X, *c, **B;
    int   count, funcount, gradcount;
    double f, gradproj;
    int   i, j, ilast, iter = 0;
    double s, steplength;
    double D1, D2;
    int   n, *l;

    if (maxit <= 0) {
	*fail = 0;
	*Fmin = fminfn(n0, b, ex);
	*fncount = *grcount = 0;
	return;
    }

    if (nREPORT <= 0)
	 PROBLEM "REPORT must be > 0 (method = \"BFGS\")" ERROR;
    l = Salloc(n0, int);
    n = 0;
    for (i = 0; i < n0; i++) if (mask[i]) l[n++] = i;

    g = vect(n0);
    t = vect(n);
    X = vect(n);
    c = vect(n);
    B = Lmatrix(n);
    f = fminfn(n, b, ex);
    if (!FINITE(f))
	 PROBLEM "initial value in vmmin is not finite" ERROR;
    if (trace) printf("initial  value %f \n", f);
    *Fmin = f;
    funcount = gradcount = 1;
    fmingr(n0, b, g, ex);
    iter++;
    ilast = gradcount;

    do {
	if (ilast == gradcount) {
	    for (i = 0; i < n; i++) {
		for (j = 0; j < i; j++) B[i][j] = 0.0;
		B[i][i] = 1.0;
	    }
	}
	for (i = 0; i < n; i++) {
	    X[i] = b[l[i]];
	    c[i] = g[l[i]];
	}
	gradproj = 0.0;
	for (i = 0; i < n; i++) {
	    s = 0.0;
	    for (j = 0; j <= i; j++) s -= B[i][j] * g[l[j]];
	    for (j = i + 1; j < n; j++) s -= B[j][i] * g[l[j]];
	    t[i] = s;
	    gradproj += s * g[l[i]];
	}

	if (gradproj < 0.0) {	/* search direction is downhill */
	    steplength = 1.0;
	    accpoint = S_FALSE;
	    do {
		count = 0;
		for (i = 0; i < n; i++) {
		    b[l[i]] = X[i] + steplength * t[i];
		    if (reltest + X[i] == reltest + b[l[i]]) /* no change */
			count++;
		}
		if (count < n) {
		    f = fminfn(n0, b, ex);
		    funcount++;
		    accpoint = FINITE(f) &&
			(f <= *Fmin + gradproj * steplength * acctol);
		    if (!accpoint) {
			steplength *= stepredn;
		    }
		}
	    } while (!(count == n || accpoint));
	    enough = (f > abstol) &&
		fabs(f - *Fmin) > reltol * (fabs(*Fmin) + reltol);
	    /* stop if value if small or if relative change is low */
	    if (!enough) {
		count = n;
		*Fmin = f;
	    }
	    if (count < n) {/* making progress */
		*Fmin = f;
		fmingr(n0, b, g, ex);
		gradcount++;
		iter++;
		D1 = 0.0;
		for (i = 0; i < n; i++) {
		    t[i] = steplength * t[i];
		    c[i] = g[l[i]] - c[i];
		    D1 += t[i] * c[i];
		}
		if (D1 > 0) {
		    D2 = 0.0;
		    for (i = 0; i < n; i++) {
			s = 0.0;
			for (j = 0; j <= i; j++)
			    s += B[i][j] * c[j];
			for (j = i + 1; j < n; j++)
			    s += B[j][i] * c[j];
			X[i] = s;
			D2 += s * c[i];
		    }
		    D2 = 1.0 + D2 / D1;
		    for (i = 0; i < n; i++) {
			for (j = 0; j <= i; j++)
			    B[i][j] += (D2 * t[i] * t[j]
					- X[i] * t[j] - t[i] * X[j]) / D1;
		    }
		} else {	/* D1 < 0 */
		    ilast = gradcount;
		}
	    } else {	/* no progress */
		if (ilast < gradcount) {
		    count = 0;
		    ilast = gradcount;
		}
	    }
	} else {		/* uphill search */
	    count = 0;
	    if (ilast == gradcount) count = n;
	    else ilast = gradcount;
	    /* Resets unless has just been reset */
	}
	if (trace && (iter % nREPORT == 0))
	    printf("iter%4d value %f\n", iter, f);
	if (iter >= maxit) break;
	if (gradcount - ilast > 2 * n)
	    ilast = gradcount;	/* periodic restart */
    } while (count != n || ilast != gradcount);
    if (trace) {
	printf("final  value %f \n", *Fmin);
	if (iter < maxit) printf("converged\n");
	else printf("stopped after %i iterations\n", iter);
    }
    *fail = (iter < maxit) ? 0 : 1;
    *fncount = funcount;
    *grcount = gradcount;
}


#define big             1.0e+35   /*a very large number*/


/* Nelder-Mead */
static void
nmmin(int n, double *Bvec, double *X, double *Fmin, optimfn fminfn,
      int *fail, double abstol, double intol, void *ex,
      double alpha, double beta, double gamm, int trace,
      int *fncount, int maxit)
{
    char action[50];
    int C;
    s_boolean calcvert, shrinkfail = S_FALSE;
    double convtol, f;
    int funcount=0, H, i, j, L=0;
    int n1=0;
    double oldsize;
    double **P;
    double size, step, temp, trystep;
    char tstr[6];
    double VH, VL, VR;

    if (maxit <= 0) {
	*Fmin = fminfn(n, Bvec, ex);
	*fncount = 0;
	*fail = 0;
	return;
    }
    if (trace)
	printf("  Nelder-Mead direct search function minimizer\n");
    P = matrix(n, n+1);
    *fail = S_FALSE;
    f = fminfn(n, Bvec, ex);
    if (!FINITE(f)) {
	 PROBLEM "Function cannot be evaluated at initial parameters" ERROR;
	*fail = S_TRUE;
    } else {
	if (trace) printf("Function value for initial parameters = %f\n", f);
	funcount = 1;
	convtol = intol * (fabs(f) + intol);
	if (trace) printf("  Scaled convergence tolerance is %g\n", convtol);
	n1 = n + 1;
	C = n + 2;
	P[n1 - 1][0] = f;
	for (i = 0; i < n; i++)
	    P[i][0] = Bvec[i];

	L = 1;
	size = 0.0;

	step = 0.0;
	for (i = 0; i < n; i++) {
	    if (0.1 * fabs(Bvec[i]) > step)
		step = 0.1 * fabs(Bvec[i]);
	}
	if (step == 0.0) step = 0.1;
	if (trace) printf("Stepsize computed as %f\n", step);
	for (j = 2; j <= n1; j++) {
	    strcpy(action, "BUILD          ");
	    for (i = 0; i < n; i++)
		P[i][j - 1] = Bvec[i];

	    trystep = step;
	    while (P[j - 2][j - 1] == Bvec[j - 2]) {
		P[j - 2][j - 1] = Bvec[j - 2] + trystep;
		trystep *= 10;
	    }
	    size += trystep;
	}
	oldsize = size;
	calcvert = S_TRUE;
	shrinkfail = S_FALSE;
	do {
	    if (calcvert) {
		for (j = 0; j < n1; j++) {
		    if (j + 1 != L) {
			for (i = 0; i < n; i++)
			    Bvec[i] = P[i][j];
			f = fminfn(n, Bvec, ex);
			if (!FINITE(f)) f = big;
			funcount++;
			P[n1 - 1][j] = f;
		    }
		}
		calcvert = S_FALSE;
	    }

	    VL = P[n1 - 1][L - 1];
	    VH = VL;
	    H = L;

	    for (j = 1; j <= n1; j++) {
		if (j != L) {
		    f = P[n1 - 1][j - 1];
		    if (f < VL) {
			L = j;
			VL = f;
		    }
		    if (f > VH) {
			H = j;
			VH = f;
		    }
		}
	    }

	    if (VH > VL + convtol && VL > abstol) {
		sprintf(tstr, "%5d", funcount);
		if (trace) printf("%s%s %f %f\n", action, tstr, VH, VL);

		for (i = 0; i < n; i++) {
		    temp = -P[i][H - 1];
		    for (j = 0; j < n1; j++)
			temp += P[i][j];
		    P[i][C - 1] = temp / n;
		}
		for (i = 0; i < n; i++)
		    Bvec[i] = (1.0 + alpha) * P[i][C - 1] -
			alpha * P[i][H - 1];
		f = fminfn(n, Bvec, ex);
		if (!FINITE(f)) f = big;
		funcount++;
		strcpy(action, "REFLECTION     ");
		VR = f;
		if (VR < VL) {
		    P[n1 - 1][C - 1] = f;
		    for (i = 0; i < n; i++) {
			f = gamm * Bvec[i] + (1 - gamm) * P[i][C - 1];
			P[i][C - 1] = Bvec[i];
			Bvec[i] = f;
		    }
		    f = fminfn(n, Bvec, ex);
		    if (!FINITE(f)) f = big;
		    funcount++;
		    if (f < VR) {
			for (i = 0; i < n; i++)
			    P[i][H - 1] = Bvec[i];
			P[n1 - 1][H - 1] = f;
			strcpy(action, "EXTENSION      ");
		    } else {
			for (i = 0; i < n; i++)
			    P[i][H - 1] = P[i][C - 1];
			P[n1 - 1][H - 1] = VR;
		    }
		} else {
		    strcpy(action, "HI-REDUCTION   ");
		    if (VR < VH) {
			for (i = 0; i < n; i++)
			    P[i][H - 1] = Bvec[i];
			P[n1 - 1][H - 1] = VR;
			strcpy(action, "LO-REDUCTION   ");
		    }

		    for (i = 0; i < n; i++)
			Bvec[i] = (1 - beta) * P[i][H - 1] +
			    beta * P[i][C - 1];
		    f = fminfn(n, Bvec, ex);
		    if (!FINITE(f)) f = big;
		    funcount++;

		    if (f < P[n1 - 1][H - 1]) {
			for (i = 0; i < n; i++)
			    P[i][H - 1] = Bvec[i];
			P[n1 - 1][H - 1] = f;
		    } else {
			if (VR >= VH) {
			    strcpy(action, "SHRINK         ");
			    calcvert = S_TRUE;
			    size = 0.0;
			    for (j = 0; j < n1; j++) {
				if (j + 1 != L) {
				    for (i = 0; i < n; i++) {
					P[i][j] = beta * (P[i][j] - P[i][L - 1]) + P[i][L - 1];
					size += fabs(P[i][j] - P[i][L - 1]);
				    }
				}
			    }
			    if (size < oldsize) {
				shrinkfail = S_FALSE;
				oldsize = size;
			    } else {
				if (trace)
				    printf("Polytope size measure not decreased in shrink\n");
				shrinkfail = S_TRUE;
			    }
			}
		    }
		}
	    }

	} while (!(VH <= VL + convtol || VL <= abstol ||
		   shrinkfail || funcount > maxit));

    }

    if (trace) {
	printf("Exiting from Nelder Mead minimizer\n");
	printf("    %d function evaluations used\n", funcount);
    }
    *Fmin = P[n1 - 1][L - 1];
    for (i = 0; i < n; i++) X[i] = P[i][L - 1];
    if (shrinkfail) *fail = 10;
    if (funcount > maxit) *fail = 1;
    *fncount = funcount;
}

static void
cgmin(int n, double *Bvec, double *X, double *Fmin,
      optimfn fminfn, optimgr fmingr, int *fail,
      double abstol, double intol, void *ex, int type, int trace,
      int *fncount, int *grcount, int maxit)
{
    s_boolean accpoint;
    double *c, *g, *t;
    int count, cycle, cyclimit;
    double f;
    double G1, G2, G3, gradproj;
    int funcount=0, gradcount=0, i;
    double newstep, oldstep, setstep, steplength=1.0;
    double tol;

    if (maxit <= 0) {
	*Fmin = fminfn(n, Bvec, ex);
	*fncount = *grcount = 0;
	*fail = 0;
	return;
    }
    if (trace) {
	printf("  Conjugate gradients function minimiser\n");
	switch (type) {
	case 1:
	    printf("Method: Fletcher Reeves\n");
	    break;
	case 2:
	    printf("Method: Polak Ribiere\n");
	    break;
	case 3:
	    printf("Method: Beale Sorenson\n");
	    break;
	default:
	     PROBLEM "unknown type in CG method of optim" ERROR;
	}
    }
    c = vect(n); g = vect(n); t = vect(n);

    setstep = 1.7;
    *fail = 0;
    cyclimit = n;
    tol = intol * n * sqrt(intol);

    if (trace) printf("tolerance used in gradient test=%g\n", tol);
    f = fminfn(n, Bvec, ex);
    if (!FINITE(f)) {
	 PROBLEM "Function cannot be evaluated at initial parameters" ERROR;
    } else {
	*Fmin = f;
	funcount = 1;
	gradcount = 0;
	do {
	    for (i = 0; i < n; i++) {
		t[i] = 0.0;
		c[i] = 0.0;
	    }
	    cycle = 0;
	    oldstep = 1.0;
	    count = 0;
	    do {
		cycle++;
		if (trace) {
		    printf("%d %d %f\n", gradcount, funcount, *Fmin);
		    printf("parameters ");
		    for (i = 1; i <= n; i++) {
			printf("%10.5f ", Bvec[i - 1]);
			if (i / 7 * 7 == i && i < n)
			    printf("\n");
		    }
		    printf("\n");
		}
		gradcount++;
		if (gradcount > maxit) {
		    *fncount = funcount;
		    *grcount = gradcount;
		    *fail = 1;
		    return;
		}
		fmingr(n, Bvec, g, ex);
		G1 = 0.0;
		G2 = 0.0;
		for (i = 0; i < n; i++) {
		    X[i] = Bvec[i];
		    switch (type) {

		    case 1: /* Fletcher-Reeves */
			G1 += g[i] * g[i];
			G2 += c[i] * c[i];
			break;

		    case 2: /* Polak-Ribiere */
			G1 += g[i] * (g[i] - c[i]);
			G2 += c[i] * c[i];
			break;

		    case 3: /* Beale-Sorenson */
			G1 += g[i] * (g[i] - c[i]);
			G2 += t[i] * (g[i] - c[i]);
			break;

		    default:
			 PROBLEM "unknown type in CG method of optim" ERROR;
		    }
		    c[i] = g[i];
		}
		if (G1 > tol) {
		    if (G2 > 0.0)
			G3 = G1 / G2;
		    else
			G3 = 1.0;
		    gradproj = 0.0;
		    for (i = 0; i < n; i++) {
			t[i] = t[i] * G3 - g[i];
			gradproj += t[i] * g[i];
		    }
		    steplength = oldstep;

		    accpoint = S_FALSE;
		    do {
			count = 0;
			for (i = 0; i < n; i++) {
			    Bvec[i] = X[i] + steplength * t[i];
			    if (reltest + X[i] == reltest + Bvec[i])
				count++;
			}
			if (count < n) {
			    f = fminfn(n, Bvec, ex);
			    funcount++;
			    accpoint = (FINITE(f) &&
					f <= *Fmin + gradproj * steplength * acctol);

			    if (!accpoint) {
				steplength *= stepredn;
				if (trace) printf("*");
			    }
			}
		    } while (!(count == n || accpoint));
		    if (count < n) {
			newstep = 2 * (f - *Fmin - gradproj * steplength);
			if (newstep > 0) {
			    newstep = -(gradproj * steplength * steplength / newstep);
			    for (i = 0; i < n; i++)
				Bvec[i] = X[i] + newstep * t[i];
			    *Fmin = f;
			    f = fminfn(n, Bvec, ex);
			    funcount++;
			    if (f < *Fmin) {
				*Fmin = f;
				if (trace) printf(" i< ");
			    } else {
				if (trace) printf(" i> ");
				for (i = 0; i < n; i++)
				    Bvec[i] = X[i] + steplength * t[i];
			    }
			}
		    }
		}
		oldstep = setstep * steplength;
		if (oldstep > 1.0)
		    oldstep = 1.0;
	    } while ((count != n) && (G1 > tol) && (cycle != cyclimit));

	} while ((cycle != 1) ||
		 ((count != n) && (G1 > tol) && *Fmin > abstol));

    }
    if (trace) {
	printf("Exiting from conjugate gradients minimizer\n");
	printf("    %d function evaluations used\n", funcount);
	printf("    %d gradient evaluations used\n", gradcount);
    }
    *fncount = funcount;
    *grcount = gradcount;
}

static
void lbfgsb(int n, int m, double *x, double *l, double *u, int *nbd,
	    double *Fmin, optimfn fminfn, optimgr fmingr, int *fail,
	    void *ex, double factr, double pgtol,
	    int *fncount, int *grcount, int maxit, char *msg,
	    int trace, int nREPORT)
{
    char task[60];
    double f, *g, dsave[29], *wa;
    int tr = -1, iter = 0, *iwa, isave[44], lsave[4];

    if (nREPORT <= 0)
	 PROBLEM "REPORT must be > 0 (method = \"L-BFGS-B\")" ERROR;
    switch(trace) {
    case 2: tr = 0; break;
    case 3: tr = nREPORT; break;
    case 4: tr = 99; break;
    case 5: tr = 100; break;
    case 6: tr = 101; break;
    default: tr = -1; break;
    }

    *fail = 0;
    g = vect(n);
    wa = vect(2*m*n+4*n+11*m*m+8*m);
    iwa = Salloc(3*n, int);
    strcpy(task, "START");
    while(1) {
	/* Main workhorse setulb() from ../appl/lbfgsb.c : */
	setulb(n, m, x, l, u, nbd, &f, g, factr, &pgtol, wa, iwa, task,
	       tr, lsave, isave, dsave);
/*	printf("in lbfgsb - %s\n", task);*/
	if (strncmp(task, "FG", 2) == 0) {
	    f = fminfn(n, x, ex);
	    if (!FINITE(f))
		 PROBLEM "L-BFGS-B needs finite values of fn" ERROR;
	    fmingr(n, x, g, ex);
	} else if (strncmp(task, "NEW_X", 5) == 0) {
	    if(trace == 1 && (iter % nREPORT == 0)) {
		printf("iter %4d value %f\n", iter, f);
	    }
	    if (++iter > maxit) {
		*fail = 1;
		break;
	    }
	} else if (strncmp(task, "WARN", 4) == 0) {
	    *fail = 51;
	    break;
	} else if (strncmp(task, "CONV", 4) == 0) {
	    break;
	} else if (strncmp(task, "ERROR", 5) == 0) {
	    *fail = 52;
	    break;
	} else { /* some other condition that is not supposed to happen */
	    *fail = 52;
	    break;
	}
    }
    *Fmin = f;
    *fncount = *grcount = isave[33];
    if (trace) {
	printf("final  value %f \n", *Fmin);
	if (iter < maxit && *fail == 0) printf("converged\n");
	else printf("stopped after %i iterations\n", iter);
    }
    strcpy(msg, task);
}
#define E1 1.7182818  /* exp(1.0)-1.0 */
#define STEPS 100
#include "verS.h"

static void
samin(int n, double *pb, double *yb, optimfn fminfn, int maxit,
      int tmax, double ti, int trace, void *ex)

/* Given a starting point pb[0..n-1], simulated annealing minimization
   is performed on the function fminfn. The starting temperature
   is input as ti. To make sann work silently set trace to zero.
   sann makes in total maxit function evaluations, tmax
   evaluations at each temperature. Returned quantities are pb
   (the location of the minimum), and yb (the minimum value of
   the function func).  Author: Adrian Trapletti
*/
{
    S_EVALUATOR
    long i, j;
    int k, its, itdoc;
    double t, y, dy, ytry, scale;
    double *p, *dp, *ptry;

    p = vect (n); dp = vect (n); ptry = vect (n);
    RANDIN;
    *yb = fminfn (n, pb, ex);  /* init best system state pb, *yb */
    if (!FINITE(*yb)) *yb = big;
    for (j = 0; j < n; j++) p[j] = pb[j];
    y = *yb;  /* init system state p, y */
    if (trace)
    {
	printf ("sann objective function values\n");
	printf ("initial       value %f\n", *yb);
    }
    scale = 1.0/ti;
    its = itdoc = 1;
    while (its < maxit) { /* cool down system */
	t = ti/log((double)its + E1);  /* temperature annealing schedule */
	k = 1;
	while ((k <= tmax) && (its < maxit))  /* iterate at constant temperature */
	{
	    for (i = 0; i < n; i++)
		dp[i] = scale * t * norm_rand(S_evaluator);  /* random perturbation */
	    for (i = 0; i < n; i++)
		ptry[i] = p[i] + dp[i];  /* new candidate point */
	    ytry = fminfn (n, ptry, ex);
	    if (!FINITE(ytry)) ytry = big;
	    dy = ytry - y;
	    if ((dy <= 0.0) || (UNIF < exp(-dy/t))) {  /* accept new point? */
		for (j = 0; j < n; j++) p[j] = ptry[j];
		y = ytry;  /* update system state p, y */
		if (y <= *yb)  /* if system state is best, then update best system state pb, *yb */
		{
		    for (j = 0; j < n; j++) pb[j] = p[j];
		    *yb = y;
		}
	    }
	    its++; k++;
	}
	if ((trace) && ((itdoc % STEPS) == 0))
	    printf("iter %8d value %f\n", its - 1, *yb);
	itdoc++;
    }
    if (trace)
    {
	printf ("final         value %f\n", *yb);
	printf ("sann stopped after %d iterations\n", its - 1);
    }
    RANDOUT;
}

#undef E1
#undef STEPS
