#include <S.h>

#ifdef WIN32
/* based on
Cephes Math Library Release 2.8:  June, 2000
Copyright (C) 1987, 1995, 2000 by Stephen L. Moshier
*/

double polevl(double x, double *coef, int N)
{
    double *p = coef, ans;
    int i = N;

    ans = *p++;
    do ans = ans * x  +  *p++; while( --i );
    return( ans );
}

double p1evl(double x, double *coef, int N)
{
    double *p = coef, ans;
    int i = N - 1;

    ans = x + *p++;
    do ans = ans * x  + *p++; while( --i );
    return( ans );
}

static double P[] = {
-8.54074331929669305196E-1,
 1.20426861384072379242E1,
-4.61252884198732692637E1,
 6.54566728676544377376E1,
-3.09092539379866942570E1
};
static double Q[] = {
/* 1.00000000000000000000E0,*/
-1.95638849376911654834E1,
 1.08938092147140262656E2,
-2.49839401325893582852E2,
 2.52006675691344555838E2,
-9.27277618139601130017E1
};

double atanh(double x)
{
    double s, z;

    if( x == 0.0 ) return x;
    z = fabs(x);
    if( z < 1.0e-7 ) return x;
    if( z < 0.5 ) {
	z = x * x;
	s = x + x * z * (polevl(z, P, 4) / p1evl(z, Q, 5));
	return s;
    }

    return 0.5 * log((1.0+x)/(1.0-x));
}
#endif

typedef  s_object * SEXP;

#define R_NilValue NEW_INTEGER(0);

#ifndef max
#define max(a,b) ((a < b)?(b):(a))
#endif
#ifndef min
#define min(a,b) ((a > b)?(b):(a))
#endif

SEXP
ARMAtoMA(SEXP ar, SEXP ma, SEXP lag_max)
{
    int i, j, p = GET_LENGTH(ar), q = GET_LENGTH(ma),
	m = INTEGER_VALUE(lag_max);
    double *phi = NUMERIC_POINTER(ar), *theta = NUMERIC_POINTER(ma),
	*psi, tmp;
    SEXP res;

    if(m <= 0) PROBLEM "invalid value of lag.max" ERROR;
    res = NEW_NUMERIC(m);
    psi = NUMERIC_POINTER(res);
    for(i = 0; i < m; i++) {
	tmp = (i < q) ? theta[i] : 0.0;
	for(j = 0; j < min(i+1, p); j++)
	    tmp += phi[j] * ((i-j-1 >= 0) ? psi[i-j-1] : 1.0);
	psi[i] = tmp;
    }
    return res;
}

/* cor is the autocorrelations starting from 0 lag*/
void uni_pacf(double *cor, double *p, int *pnlag)
{
    S_EVALUATOR
    int nlag = *pnlag;
    int i, ll;
    double a, b, c, *v, *w;

    v = Salloc(nlag, double);
    w = Salloc(nlag, double);
    w[0] = p[0] = cor[1];
    for(ll = 1; ll < nlag; ll++) {
	a = cor[ll+1];
	b = 1.0;
	for(i = 0; i < ll; i++) {
	    a -= w[i] * cor[ll - i];
	    b -= w[i] * cor[i + 1];
	}
	p[ll] = c = a/b;
	if(ll+1 == nlag) break;
	w[ll] = c;
	for(i = 0; i < ll; i++)
	    v[ll-i-1] = w[i];
	for(i = 0; i < ll; i++)
	    w[i] -= c*v[i];
    }
}


typedef struct
{
    int p, q, r, np, nrbar, n, ncond, m, trans, method, nused;
    int mp, mq, msp, msq, ns;
    double delta, s2;
    double *params, *phi, *theta, *a, *P, *V;
    double *thetab, *xnext, *xrow, *rbar, *w, *wkeep, *resid, *reg;
} starma_struct, *Starma;

static void starma(Starma G, int *ifault);

static void karma(Starma G, double *sumlog, double *ssq, int iupd, int *nit);

static void forkal(Starma G, int id, int il, double *delta, double *y,
		   double *amse, int *ifault);

static Starma G;

/* Internal */
static void partrans(int np, double *raw, double *new);
static void dotrans(Starma G, double *raw, double *new, int trans);

SEXP setup_starma(SEXP na, SEXP x, SEXP pn, SEXP xreg, SEXP pm,
		  SEXP dt, SEXP ptrans, SEXP sncond)
{
    S_EVALUATOR
    int i, n, m, ip, iq, ir, np;

    G = Calloc(1, starma_struct);
    G->mp = INTEGER_POINTER(na)[0];
    G->mq = INTEGER_POINTER(na)[1];
    G->msp = INTEGER_POINTER(na)[2];
    G->msq = INTEGER_POINTER(na)[3];
    G->ns = INTEGER_POINTER(na)[4];
    G->n = n = INTEGER_VALUE(pn);
    G->ncond = INTEGER_VALUE(sncond);
    G->m = m = INTEGER_VALUE(pm);
    G->params = Calloc(G->mp + G->mq + G->msp + G->msq + G->m, double);
    G->p = ip = G->ns*G->msp + G->mp;
    G->q = iq = G->ns*G->msq + G->mq;
    G->r = ir = max(ip, iq + 1);
    G->np = np = (ir*(ir + 1))/2;
    G->nrbar = max(1, np*(np - 1)/2);
    G->trans = INTEGER_VALUE(ptrans);
    G->a = Calloc(ir, double);
    G->P = Calloc(np, double);
    G->V = Calloc(np, double);
    G->thetab = Calloc(np, double);
    G->xnext = Calloc(np, double);
    G->xrow = Calloc(np, double);
    G->rbar = Calloc(G->nrbar, double);
    G->w = Calloc(n, double);
    G->wkeep = Calloc(n, double);
    G->resid = Calloc(n, double);
    G->phi = Calloc(ir, double);
    G->theta = Calloc(ir, double);
    G->reg = Calloc(1 + n*m, double);
    G->delta = NUMERIC_VALUE(dt);
    x = AS_NUMERIC(x);
    xreg = AS_NUMERIC(xreg);
    for(i = 0; i < n; i++) G->w[i] = G->wkeep[i] = NUMERIC_POINTER(x)[i];
    for(i = 0; i < n*m; i++) G->reg[i] = NUMERIC_POINTER(xreg)[i];
    return R_NilValue;
}

SEXP free_starma(void)
{
    S_EVALUATOR
    Free(G->params); Free(G->a); Free(G->P); Free(G->V); Free(G->thetab);
    Free(G->xnext); Free(G->xrow); Free(G->rbar);
    Free(G->w); Free(G->wkeep); Free(G->resid); Free(G->phi); Free(G->theta);
    Free(G->reg); Free(G);
    return R_NilValue;
}

SEXP Starma_method(SEXP method)
{
    S_EVALUATOR
    G->method = INTEGER_VALUE(method);
    return R_NilValue;
}

SEXP Dotrans(SEXP x)
{
    S_EVALUATOR
    SEXP y = NEW_NUMERIC(LENGTH(x));

    dotrans(G, NUMERIC_POINTER(x), NUMERIC_POINTER(y), 1);
    return y;
}

SEXP set_trans(SEXP ptrans)
{
    S_EVALUATOR
    G->trans = INTEGER_VALUE(ptrans);
    return R_NilValue;
}

SEXP arma0fa(SEXP inparams)
{
    S_EVALUATOR
    int i, j, ifault = 0, it, streg;
    double sumlog, ssq, tmp, ans;
    SEXP res;

    dotrans(G, NUMERIC_POINTER(inparams), G->params, G->trans);

    if(G->ns > 0) {
	/* expand out seasonal ARMA models */
	for(i = 0; i < G->mp; i++) G->phi[i] = G->params[i];
	for(i = 0; i < G->mq; i++) G->theta[i] = G->params[i + G->mp];
	for(i = G->mp; i < G->p; i++) G->phi[i] = 0.0;
	for(i = G->mq; i < G->q; i++) G->theta[i] = 0.0;
	for(j = 0; j < G->msp; j++) {
	    G->phi[(j + 1)*G->ns - 1] += G->params[j + G->mp + G->mq];
	    for(i = 0; i < G->mp; i++)
		G->phi[(j + 1)*G->ns + i] -= G->params[i]*
		    G->params[j + G->mp + G->mq];
	}
	for(j = 0; j < G->msq; j++) {
	    G->theta[(j + 1)*G->ns - 1] +=
		G->params[j + G->mp + G->mq + G->msp];
	    for(i = 0; i < G->mq; i++)
		G->theta[(j + 1)*G->ns + i] += G->params[i + G->mp]*
		    G->params[j + G->mp + G->mq + G->msp];
	}
    } else {
	for(i = 0; i < G->mp; i++) G->phi[i] = G->params[i];
	for(i = 0; i < G->mq; i++) G->theta[i] = G->params[i + G->mp];
    }

    streg = G->mp + G->mq + G->msp + G->msq;
    if(G->m > 0) {
	for(i = 0; i < G->n; i++) {
	    tmp = G->wkeep[i];
	    for(j = 0; j < G->m; j++)
		tmp -= G->reg[i + G->n*j] * G->params[streg + j];
	    G->w[i] = tmp;
	}
    }

    if(G->method == 1) {
	int p = G->mp + G->ns * G->msp, q = G->mq + G->ns * G->msq, nu = 0;
	ssq = 0.0;
	for(i = 0; i < G->ncond; i++) G->resid[i] = 0.0;
	for(i = G->ncond; i < G->n; i++) {
	    tmp = G->w[i];
	    for(j = 0; j < p; j++)
		tmp -= G->phi[j] * G->wkeep[i - j - 1];
	    for(j = 0; j < min(i - G->ncond, q); j++)
		tmp -= G->theta[j] * G->resid[i - j - 1];
	    G->resid[i] = tmp;
	    if(!is_nan_DOUBLE(&tmp)) {
		nu++;
		ssq += tmp * tmp;
	    }
	}
	G->s2 = ssq/(double)(nu);
	ans = 0.5 * log(G->s2);
    } else {
	starma(G, &ifault);
	if(ifault) PROBLEM "starma error code %d", ifault ERROR;
	sumlog = 0.0;
	ssq = 0.0;
	it = 0;
	karma(G, &sumlog, &ssq, 1, &it);
	G->s2 = ssq/(double)G->nused;
	ans = 0.5*(log(ssq/(double)G->nused) + sumlog/(double)G->nused);
    }
    res = NEW_NUMERIC(1);
    NUMERIC_POINTER(res)[0] = ans;
    return res;
}

SEXP get_s2(void)
{
    S_EVALUATOR
    SEXP res = NEW_NUMERIC(1);

    NUMERIC_POINTER(res)[0] = G->s2;
    return res;
}

SEXP get_resid(void)
{
    S_EVALUATOR
    SEXP res;
    int i;

    res = NEW_NUMERIC(G->n);
    for(i = 0; i < G->n; i++) NUMERIC_POINTER(res)[i] = G->resid[i];
    return res;
}

SEXP arma0_kfore(SEXP pd, SEXP psd, SEXP nahead)
{
    S_EVALUATOR
    int dd = INTEGER_VALUE(pd);
    int d, il = INTEGER_VALUE(nahead), ifault = 0, i, j;
    double *del, *del2;
    SEXP res, x, var;

    res = NEW_LIST(2);
    SET_ELEMENT(res, 0, x = NEW_NUMERIC(il));
    SET_ELEMENT(res, 1, var = NEW_NUMERIC(il));

    d = dd + G->ns * INTEGER_VALUE(psd);

    del = Salloc(d + 1, double);
    del2 = Salloc(d + 1, double);
    del[0] = 1;
    for(i = 1; i <= d; i++) del[i] = 0;
    for (j = 0; j < dd; j++) {
	for(i = 0; i <= d; i++) del2[i] = del[i];
	for(i = 0; i <= d - 1; i++) del[i+1] -= del2[i];
    }
    for (j = 0; j < INTEGER_VALUE(psd); j++) {
	for(i = 0; i <= d; i++) del2[i] = del[i];
	for(i = 0; i <= d - G->ns; i++) del[i + G->ns] -= del2[i];
    }
    for(i = 1; i <= d; i++) del[i] *= -1;


    forkal(G, d, il, del + 1, NUMERIC_POINTER(x), NUMERIC_POINTER(var),
	   &ifault);
    if(ifault) PROBLEM "forkal error code %d", ifault ERROR;
    return res;
}

static void partrans(int p, double *raw, double *new)
{
    int j, k;
    double a, work[100];

    if(p > 100) PROBLEM "can only transform 100 pars in arima" ERROR;

    /* Step one: map (-Inf, Inf) to (-1, 1) via tanh
       The parameters are now the pacf phi_{kk} */
    for(j = 0; j < p; j++) work[j] = new[j] = tanh(raw[j]);
    /* Step two: run the Durbin-Levinson recursions to find phi_{j.},
       j = 2, ..., p and phi_{p.} are the autoregression coefficients */
    for(j = 1; j < p; j++) {
	a = new[j];
	for(k = 0; k < j; k++)
	    work[k] -= a * new[j - k - 1];
	for(k = 0; k < j; k++) new[k] = work[k];
    }
}

static void dotrans(Starma G, double *raw, double *new, int trans)
{
    int i, v, n = G->mp + G->mq + G->msp + G->msq + G->m;

    for(i = 0; i < n; i++) new[i] = raw[i];
    if(trans) {
	partrans(G->mp, raw, new);
	v = G->mp + G->mq;
	partrans(G->msp, raw + v, new + v);
    }
}
static void invpartrans(int p, double *phi, double *new)
{
    int j, k;
    double a, work[100];

    if(p > 100) PROBLEM "can only transform 100 pars in arima" ERROR;

    for(j = 0; j < p; j++) work[j] = new[j] = phi[j];
    /* Run the Durbin-Levinson recursions backwards
       to find the PACF phi_{j.} from the autoregression coefficients */
    for(j = p - 1; j > 0; j--) {
	a = new[j];
	for(k = 0; k < j; k++)
	    work[k]  = (new[k] + a * new[j - k - 1]) / (1 - a * a);
	for(k = 0; k < j; k++) new[k] = work[k];
    }
    for(j = 0; j < p; j++) new[j] = atanh(new[j]);
}

SEXP Invtrans(SEXP x)
{
    SEXP y = NEW_NUMERIC(LENGTH(x));
    int i, v, n;
    double *raw = NUMERIC_POINTER(x), *new = NUMERIC_POINTER(y);

    n = G->mp + G->mq + G->msp + G->msq + G->m;

    for(i = 0; i < n; i++) new[i] = raw[i];
    invpartrans(G->mp, raw, new);
    v = G->mp + G->mq;
    invpartrans(G->msp, raw + v, new + v);
    return y;
}

#define eps 1e-3
SEXP Gradtrans(SEXP x)
{
    SEXP y = NEW_NUMERIC(LENGTH(x) * LENGTH(x));
    int i, j, v, n;
    double *raw = NUMERIC_POINTER(x), *A = NUMERIC_POINTER(y),
	w1[100], w2[100], w3[100];

    n = G->mp + G->mq + G->msp + G->msq + G->m;
    for(i = 0; i < n; i++)
	for(j = 0; j < n; j++)
	    A[i + j*n] = (i == j);
    if(G->mp > 0) {
	for(i = 0; i < G->mp; i++) w1[i] = raw[i];
	partrans(G->mp, w1, w2);
	for(i = 0; i < G->mp; i++) {
	    w1[i] += eps;
	    partrans(G->mp, w1, w3);
	    for(j = 0; j < G->mp; j++) A[i + j*n] = (w3[j] - w2[j])/eps;
	    w1[i] -= eps;
	}
    }
    if(G->mq > 0) {
	v = G->mp;
	for(i = 0; i < G->mq; i++) w1[i] = raw[i + v];
	partrans(G->mq, w1, w2);
	for(i = 0; i < G->mq; i++) {
	    w1[i] += eps;
	    partrans(G->mq, w1, w3);
	    for(j = 0; j < G->mq; j++) A[i + v + j*n] = (w3[j] - w2[j])/eps;
	    w1[i] -= eps;
	}
    }
    if(G->msp > 0) {
	v = G->mp + G->mq;
	for(i = 0; i < G->msp; i++) w1[i] = raw[i + v];
	partrans(G->msp, w1, w2);
	for(i = 0; i < G->msp; i++) {
	    w1[i] += eps;
	    partrans(G->msp, w1, w3);
	    for(j = 0; j < G->msp; j++)
		A[i + v + (j+v)*n] = (w3[j] - w2[j])/eps;
	    w1[i] -= eps;
	}
    }
    if(G->msq > 0) {
	v = G->mp + G->mq + G->msp;
	for(i = 0; i < G->msq; i++) w1[i] = raw[i + v];
	partrans(G->msq, w1, w2);
	for(i = 0; i < G->msq; i++) {
	    w1[i] += eps;
	    partrans(G->msq, w1, w3);
	    for(j = 0; j < G->msq; j++)
		A[i + v + (j+v)*n] = (w3[j] - w2[j])/eps;
	    w1[i] -= eps;
	}
    }
    return y;
}

#ifndef max
#define max(a,b) ((a < b)?(b):(a))
#endif
#ifndef min
#define min(a,b) ((a > b)?(b):(a))
#endif

/* Code in the rest of this file is based on Applied Statistics
   algorithms AS154/182 (C) Royal Statistical Society 1980, 1982 */

static void
inclu2(int np, double *xnext, double *xrow, double ynext,
       double *d, double *rbar, double *thetab)
{
    double cbar, sbar, di, xi, xk, rbthis, dpi;
    int i, k, ithisr;

/*   This subroutine updates d, rbar, thetab by the inclusion
     of xnext and ynext. */

    for (i = 0; i < np; i++) xrow[i] = xnext[i];

    for (ithisr = 0, i = 0; i < np; i++) {
	if (xrow[i] != 0.0) {
	    xi = xrow[i];
	    di = d[i];
	    dpi = di + xi * xi;
	    d[i] = dpi;
	    cbar = di / dpi;
	    sbar = xi / dpi;
	    for (k = i + 1; k < np; k++) {
		xk = xrow[k];
		rbthis = rbar[ithisr];
		xrow[k] = xk - xi * rbthis;
		rbar[ithisr++] = cbar * rbthis + sbar * xk;
	    }
	    xk = ynext;
	    ynext = xk - xi * thetab[i];
	    thetab[i] = cbar * thetab[i] + sbar * xk;
	    if (di == 0.0) return;
	} else ithisr = ithisr + np - i - 1;
    }
}

static void
starma(Starma G, int *ifault)
{
    S_EVALUATOR
    int p = G->p, q = G->q, r = G->r, np = G->np, nrbar = G->nrbar;
    double *phi = G->phi, *theta = G->theta, *a = G->a,
	*P = G->P, *V = G->V, *thetab = G->thetab, *xnext = G->xnext,
	*xrow = G->xrow, *rbar = G->rbar;
    int indi, indj, indn;
    double phii, phij, ynext, vj, bi;
    int i, j, k, ithisr, ind, npr, ind1, ind2, npr1, im, jm;

/*      Invoking this subroutine sets the values of v and phi, and
        obtains the initial values of a and p. */

/*     Check if ar(1) */

    if (!(q > 0 || p > 1)) {
	V[0] = 1.0;
	a[0] = 0.0;
	P[0] = 1.0 / (1.0 - phi[0] * phi[0]);
	return;
    }

/*        Check for failure indication. */
    *ifault = 0;
    if (p < 0) *ifault = 1;
    if (q < 0) *ifault += 2;
    if (p == 0 && q == 0) *ifault = 4;
    k = q + 1;
    if (k < p) k = p;
    if (r != k) *ifault = 5;
    if (np != r * (r + 1) / 2) *ifault = 6;
    if (nrbar != np * (np - 1) / 2) *ifault = 7;
    if (r == 1) *ifault = 8;
    if (*ifault != 0) return;

/*        Now set a(0), V and phi. */

    for (i = 1; i < r; i++) {
	a[i] = 0.0;
	if (i >= p) phi[i] = 0.0;
	V[i] = 0.0;
	if (i < q + 1) V[i] = theta[i - 1];
    }
    a[0] = 0.0;
    if (p == 0) phi[0] = 0.0;
    V[0] = 1.0;
    ind = r;
    for (j = 1; j < r; j++) {
	vj = V[j];
	for (i = j; i < r; i++) V[ind++] = V[i] * vj;
    }

/*        Now find p(0). */

    if (p > 0) {
/*      The set of equations s * vec(p(0)) = vec(v) is solved for
        vec(p(0)).  s is generated row by row in the array xnext.  The
        order of elements in p is changed, so as to bring more leading
        zeros into the rows of s. */

	for (i = 0; i < nrbar; i++) rbar[i] = 0.0;
	for (i = 0; i < np; i++) {
	    P[i] = 0.0;
	    thetab[i] = 0.0;
	    xnext[i] = 0.0;
	}
	ind = 0;
	ind1 = -1;
	npr = np - r;
	npr1 = npr + 1;
	indj = npr;
	ind2 = npr - 1;
	for (j = 0; j < r; j++) {
	    phij = phi[j];
	    xnext[indj++] = 0.0;
	    indi = npr1 + j;
	    for (i = j; i < r; i++) {
		ynext = V[ind++];
		phii = phi[i];
		if (j != r - 1) {
		    xnext[indj] = -phii;
		    if (i != r - 1) {
			xnext[indi] -= phij;
			xnext[++ind1] = -1.0;
		    }
		}
		xnext[npr] = -phii * phij;
		if (++ind2 >= np) ind2 = 0;
		xnext[ind2] += 1.0;
		inclu2(np, xnext, xrow, ynext, P, rbar, thetab);
		xnext[ind2] = 0.0;
		if (i != r - 1) {
		    xnext[indi++] = 0.0;
		    xnext[ind1] = 0.0;
		}
	    }
	}

	ithisr = nrbar - 1;
	im = np - 1;
	for (i = 0; i < np; i++) {
	    bi = thetab[im];
	    for (jm = np - 1, j = 0; j < i; j++)
		bi -= rbar[ithisr--] * P[jm--];
	    P[im--] = bi;
	}

/*        now re-order p. */

	ind = npr;
	for (i = 0; i < r; i++) xnext[i] = P[ind++];
	ind = np - 1;
	ind1 = npr - 1;
	for (i = 0; i < npr; i++) P[ind--] = P[ind1--];
	for (i = 0; i < r; i++) P[i] = xnext[i];
    } else {

/* P(0) is obtained by backsubstitution for a moving average process. */

	indn = np;
	ind = np;
	for (i = 0; i < r; i++)
	    for (j = 0; j <= i; j++) {
		--ind;
		P[ind] = V[ind];
		if (j != 0) P[ind] += P[--indn];
	    }
    }
}

static void
karma(Starma G, double *sumlog, double *ssq, int iupd, int *nit)
{
    S_EVALUATOR
    int p = G->p, q = G->q, r = G->r, n = G->n, nu = 0;
    double *phi = G->phi, *theta = G->theta, *a = G->a, *P = G->P,
	*V = G->V, *w = G->w, *resid = G->resid, *work = G->xnext;

    int i, j, l, ii, ind, indn, indw;
    double a1, dt, et, ft, g, ut, phij, phijdt;

/*  Invoking this subroutine updates a, P, sumlog and ssq by inclusion
    of data values w(1) to w(n). the corresponding values of resid are
    also obtained.  When ft is less than (1 + delta), quick recursions
    are used. */

/*        for non-zero values of nit, perform quick recursions. */

    if (*nit == 0) {
	for (i = 0; i < n; i++) {

/*        prediction. */

	    if (iupd != 1 || i > 0) {

/*        here dt = ft - 1.0 */

		dt = (r > 1) ? P[r] : 0.0;
		if (dt < G->delta) goto L610;
		a1 = a[0];
		for (j = 0; j < r - 1; j++) a[j] = a[j + 1];
		a[r - 1] = 0.0;
		for (j = 0; j < p; j++) a[j] += phi[j] * a1;
		if(P[0] == 0.0) { /* last obs was available */
		    ind = -1;
		    indn = r;
		    for (j = 0; j < r; j++)
			for (l = j; l < r; l++) {
			    ++ind;
			    P[ind] = V[ind];
			    if (l < r - 1) P[ind] += P[indn++];
			}
		} else {
		    for (j = 0; j < r; j++) work[j] = P[j];
		    ind = -1;
		    indn = r;
		    dt = P[0];
		    for (j = 0; j < r; j++) {
			phij = phi[j];
			phijdt = phij * dt;
			for(l = j; l < r; l++) {
			    ++ind;
			    P[ind] = V[ind] + phi[l] * phijdt;
			    if (j < r - 1) P[ind] += work[j+1] * phi[l];
			    if (l < r - 1)
				P[ind] += work[l+1] * phij + P[indn++];
			}
		    }
		}
	    }

/*        updating. */

	    ft = P[0];
	    if(!is_na_DOUBLE(&(w[i]))) {
		ut = w[i] - a[0];
		if (r > 1)
		    for (j = 1, ind = r; j < r; j++) {
			g = P[j] / ft;
			a[j] += g * ut;
			for (l = j; l < r; l++) P[ind++] -= g * P[l];
		    }
		a[0] = w[i];
		resid[i] = ut / sqrt(ft);
		*ssq += ut * ut / ft;
		*sumlog += log(ft);
		nu++;
		for (l = 0; l < r; l++) P[l] = 0.0;
	    } else na_set(&(resid[i]), S_MODE_DOUBLE);

	}
	*nit = n;

    } else {

/*        quick recursions: never used with missing values */

	i = 0;
 L610:
	*nit = i;
	for (ii = i; ii < n; ii++) {
	    et = w[ii];
	    indw = ii;
	    for (j = 0; j < p; j++) {
		if (--indw < 0) break;
		et -= phi[j] * w[indw];
	    }
	    for (j = 0; j < min(ii, q); j++)
		et -= theta[j] * resid[ii - j - 1];
	    resid[ii] = et;
	    *ssq += et * et;
	    nu++;
	}
    }
    G->nused = nu;
}


/*  start of AS 182 */
static void
forkal(Starma G, int d, int il, double *delta, double *y, double *amse,
       int *ifault)
{
    S_EVALUATOR
    int p = G->p, q = G->q, r = G->r, n = G->n, np = G->np;
    double *phi = G->phi, *V = G->V, *w = G->w, *xrow = G->xrow;
    double *a, *P, *store;
    int rd = r + d, rz = rd*(rd + 1)/2;
    double phii, phij, sigma2, a1, aa, dt, phijdt, ams, tmp;
    int i, j, k, l, nu = 0;
    int k1;
    int i45, jj, kk, lk, ll;
    int nt;
    int kk1, lk1;
    int ind, jkl, kkk;
    int ind1, ind2;

/*  Finite sample prediction from ARIMA processes. */

/*  This routine will calculate the finite sample predictions
    and their conditional mean square errors for any ARIMA process. */

/*     invoking this routine will calculate the finite sample predictions */
/*     and their conditional mean square errors for any arima process. */

    store = Salloc(rd, double);
    Free(G->a); G->a = a = Calloc(rd, double);
    Free(G->P); G->P = P = Calloc(rz, double);

/*     check for input faults. */
    *ifault = 0;
    if (p < 0) *ifault = 1;
    if (q < 0) *ifault += 2;
    if (p * p + q * q == 0) *ifault = 4;
    if (r != max(p, q + 1)) *ifault = 5;
    if (np != r * (r + 1) / 2) *ifault = 6;
    if (d < 0) *ifault = 8;
    if (il < 1) *ifault = 11;
    if (*ifault != 0) return;

/*     Find initial likelihood conditions. */

    if (r == 1) {
	a[0] = 0.0;
	V[0] = 1.0;
	P[0] = 1.0 / (1.0 - phi[0] * phi[0]);
    } else starma(G, ifault);

/*     Calculate data transformations */

    nt = n - d;
    if (d > 0) {
	for (j = 0; j < d; j++) {
	    store[j] = w[n - j - 2];
	    if(is_na_DOUBLE(store+j))
		PROBLEM "missing value in last %d observations", d ERROR;
	}
	for (i = 0; i < nt; i++) {
	    aa = 0.0;
	    for (k = 0; k < d; ++k) aa -= delta[k] * w[d + i - k - 1];
	    w[i] = w[i + d] + aa;
	}
    }

/*     Evaluate likelihood to obtain final Kalman filter conditions */

    {
	double sumlog = 0.0, ssq = 0.0;
	int nit = 0;
	G->n = nt;
	karma(G, &sumlog, &ssq, 1, &nit);
    }


/*     Calculate m.l.e. of sigma squared */

    sigma2 = 0.0;
    for (j = 0; j < nt; j++) {
	tmp = G->resid[j];
	if(!is_na_DOUBLE(&tmp)) {
	    nu++; sigma2 += tmp * tmp;
	}
    }
    sigma2 /= nu;

/*     reset the initial a and P when differencing occurs */

    if (d > 0) {
	for (i = 0; i < np; i++) xrow[i] = P[i];
	for (i = 0; i < rz; i++) P[i] = 0.0;
	ind = 0;
	for (j = 0; j < r; j++) {
	    k = j * (rd + 1) - j * (j + 1) / 2;
	    for (i = j; i < r; i++) P[k++] = xrow[ind++];
	}
	for (j = 0; j < d; j++) a[r + j] = store[j];
    }

    i45 = 2*rd + 1;
    jkl = r * (2*d + r + 1) / 2;

    for (l = 0; l < il; ++l) {

/*     predict a */

	a1 = a[0];
	for (i = 0; i < r - 1; i++) a[i] = a[i + 1];
	a[r - 1] = 0.0;
	for (j = 0; j < p; j++) a[j] += phi[j] * a1;
	if (d > 0) {
	    for (j = 0; j < d; j++) a1 += delta[j] * a[r + j];
	    for (i = rd - 1; i > r; i--) a[i] = a[i - 1];
	    a[r] = a1;
	}

/*     predict P */

	if (d > 0) {
	    for (i = 0; i < d; i++) {
		store[i] = 0.0;
		for (j = 0; j < d; j++) {
		    ll = max(i, j);
		    k = min(i, j);
		    jj = jkl + (ll - k) + k * (2*d + 2 - k - 1) / 2;
		    store[i] += delta[j] * P[jj];
		}
	    }
	    if (d > 1) {
		for (j = 0; j < d - 1; j++) {
		    jj = d - j - 1;
		    lk = (jj - 1) * (2*d + 2 - jj) / 2 + jkl;
		    lk1 = jj * (2*d + 1 - jj) / 2 + jkl;
		    for (i = 0; i <= j; i++) P[lk1++] = P[lk++];
		}
		for (j = 0; j < d - 1; j++)
		    P[jkl + j + 1] = store[j] + P[r + j];
	    }
	    P[jkl] = P[0];
	    for (i = 0; i < d; i++)
		P[jkl] += delta[i] * (store[i] + 2.0 * P[r + i]);
	    for (i = 0; i < d; i++) store[i] = P[r + i];
	    for (j = 0; j < r; j++) {
		kk1 = (j+1) * (2*rd - j - 2) / 2 + r;
		k1 = j * (2*rd - j - 1) / 2 + r;
		for (i = 0; i < d; i++) {
		    kk = kk1 + i;
		    k = k1 + i;
		    P[k] = phi[j] * store[i];
		    if (j < r - 1) P[k] += P[kk];
		}
	    }

	    for (j = 0; j < r; j++) {
		store[j] = 0.0;
		kkk = (j + 1) * (i45 - j - 1) / 2 - d;
		for (i = 0; i < d; i++) store[j] += delta[i] * P[kkk++];
	    }
	    for (j = 0; j < r; j++) {
		k = (j + 1) * (rd + 1) - (j + 1) * (j + 2) / 2;
		for (i = 0; i < d - 1; i++) {
		    --k;
		    P[k] = P[k - 1];
		}
	    }
	    for (j = 0; j < r; j++) {
		k = j * (2*rd - j - 1) / 2 + r;
		P[k] = store[j] + phi[j] * P[0];
		if (j < r - 1) P[k] += P[j + 1];
	    }
	}
	for (i = 0; i < r; i++) store[i] = P[i];

	ind = 0;
	dt = P[0];
	for (j = 0; j < r; j++) {
	    phij = phi[j];
	    phijdt = phij * dt;
	    ind2 = j * (2*rd - j + 1) / 2 - 1;
	    ind1 = (j + 1) * (i45 - j - 1) / 2 - 1;
	    for (i = j; i < r; i++) {
		++ind2;
		phii = phi[i];
		P[ind2] = V[ind++] + phii * phijdt;
		if (j < r - 1) P[ind2] += store[j + 1] * phii;
		if (i < r - 1)
		    P[ind2] += store[i + 1] * phij + P[++ind1];
	    }
	}

/*     predict y */

	y[l] = a[0];
	for (j = 0; j < d; j++) y[l] += a[r + j] * delta[j];

/*     calculate m.s.e. of y */

	ams = P[0];
	if (d > 0) {
	    for (j = 0; j < d; j++) {
		k = r * (i45 - r) / 2 + j * (2*d + 1 - j) / 2;
		tmp = delta[j];
		ams += 2.0 * tmp * P[r + j] + P[k] * tmp * tmp;
	    }
	    for (j = 0; j < d - 1; j++) {
		k = r * (i45 - r) / 2 + 1 + j * (2*d + 1 - j) / 2;
		for (i = j + 1; i < d; i++)
		    ams += 2.0 * delta[i] * delta[j] * P[k++];
	    }
	}
	amse[l] = ams * sigma2;
    }
    return;
}
