/*
 *  MASS/src/MASS.c by W. N. Venables and B. D. Ripley  Copyright (C) 1994-2003
 */

#include <stdio.h>
#include <math.h>
#include <S.h>

#if defined(SPLUS_VERSION) && SPLUS_VERSION >= 4000 && SPLUS_VERSION < 5000
#  include <newredef.h>
#endif

#ifdef USING_R
#  include "R_ext/PrtUtil.h"
#  define printf Rprintf
#endif

#include "verS.h"

#ifndef max
#  define max(a,b) ((a) > (b) ? (a) : (b))
#  define min(a,b) ((a) < (b) ? (a) : (b))
#endif

#define abs9(a) (a > 0 ? a:-a)

/* -----------------------------------------------------------------
 *  Former sammon.c
 */

void
VR_sammon(double *dd, Sint *nn, Sint *kd, double *Y, Sint *niter,
	  double *stress, Sint *trace, double *aa, double *tol)
{
    int   i, j, k, m, n = *nn, nd = *kd;
    double *xu, *xv, *e1, *e2;
    double dpj, dq, dr, dt;
    double xd, xx;
    double e, epast, eprev, tot, d, d1, ee, magic = *aa;

    xu = Calloc(nd * n, double);
    xv = Calloc(nd, double);
    e1 = Calloc(nd, double);
    e2 = Calloc(nd, double);

    epast = eprev = 1.0;
    magic = magic;

    /* Error in distances */
    e = tot = 0.0;
    for (j = 1; j < n; j++)
	for (k = 0; k < j; k++) {
	    d = dd[k * n + j];
	    if (d <= 0.0)
		PROBLEM "%s", "some distance is zero or negative"
		    RECOVER(NULL_ENTRY);
	    tot += d;
	    d1 = 0.0;
	    for (m = 0; m < nd; m++) {
		xd = Y[j + m * n] - Y[k + m * n];
		d1 += xd * xd;
	    }
	    ee = d - sqrt(d1);
	    e += (ee * ee / d);
	}
    e /= tot;
    if (*trace) {
	printf("Initial stress        : %7.5f\n", e);
	fflush(stdout);
    }
    epast = eprev = e;

    /* Iterate */
    for (i = 1; i <= *niter; i++) {
CORRECT:
	for (j = 0; j < n; j++) {
	    for (m = 0; m < nd; m++)
		e1[m] = e2[m] = 0.0;
	    for (k = 0; k < n; k++) {
		if (j == k)
		    continue;
		d1 = 0.0;
		for (m = 0; m < nd; m++) {
		    xd = Y[j + m * n] - Y[k + m * n];
		    d1 += xd * xd;
		    xv[m] = xd;
		}
		dpj = sqrt(d1);

		/* Calculate derivatives */
		dt = dd[k * n + j];
		dq = dt - dpj;
		dr = dt * dpj;
		for (m = 0; m < nd; m++) {
		    e1[m] += xv[m] * dq / dr;
		    e2[m] += (dq - xv[m] * xv[m] * (1.0 + dq / dpj) / dpj) / dr;
		}
	    }
	    /* Correction */
	    for (m = 0; m < nd; m++)
		xu[j + m * n] = Y[j + m * n] + magic * e1[m] / fabs(e2[m]);
	}

	/* Error in distances */
	e = 0.0;
	for (j = 1; j < n; j++)
	    for (k = 0; k < j; k++) {
		d = dd[k * n + j];
		d1 = 0.0;
		for (m = 0; m < nd; m++) {
		    xd = xu[j + m * n] - xu[k + m * n];
		    d1 += xd * xd;
		}
		ee = d - sqrt(d1);
		e += (ee * ee / d);
	    }
	e /= tot;
	if (e > eprev) {
	    e = eprev;
	    magic = magic * 0.2;
	    if (magic > 1.0e-3) goto CORRECT;
	    if (*trace) {
		printf("stress after %3d iters: %7.5f\n", i - 1, e);
		fflush(stdout);
	    }
	    break;
	}
	magic *= 1.5;
	if (magic > 0.5) magic = 0.5;
	eprev = e;

	/* Move the centroid to origin and update */
	for (m = 0; m < nd; m++) {
	    xx = 0.0;
	    for (j = 0; j < n; j++)
		xx += xu[j + m * n];
	    xx /= n;
	    for (j = 0; j < n; j++)
		Y[j + m * n] = xu[j + m * n] - xx;
	}

	if (i % 10 == 0) {
	    if (*trace) {
		printf("stress after %3d iters: %7.5f, magic = %5.3f\n", i, e, magic);
		fflush(stdout);
	    }
	    if (e > epast - *tol)
		break;
	    epast = e;
	}
    }
    *stress = e;
    Free(xu);
    Free(xv);
    Free(e1);
    Free(e2);
}

/*
 * ----------------------------------------------------------
 *  Former isoMDS.c

    C code for mds S-Plus library, which implements Kruskal's MDS.
    (c) B.D. Ripley, May 1995.
 *
 */

static Sint *ord;		/* ranks of dissimilarities */
static Sint *ord2;		/* inverse ordering (which one is rank i?) */
static Sint n;			/* number of  dissimilarities */
static Sint nr;			/* number of data points */
static Sint nc;			/* # cols of  fitted configuration */
static int dimx;		/* Size of configuration array */
static double *x;		/* configuration */
static double *d;		/* dissimilarities */
static double *y;		/* fitted distances (in rank of d order) */
static double *yf;		/* isotonic regression fitted values (ditto) */

void
VR_mds_fn(double *, double *, Sint *, double *, Sint *,
	  double *, Sint *, Sint *, double *, Sint *);
static void vmmin(int, double *, double *, int, int, double);

/*
 *  Download the data.
 */
void
VR_mds_init_data(Sint *pn, Sint *pc, Sint *pr, Sint *orde,
		 Sint *ordee, double *xx)
{
    int   i;

    n = *pn;
    nr = *pr;
    nc = *pc;
    dimx = nr * nc;
    ord = Calloc(n, Sint);
    ord2 = Calloc(n, Sint);
    x = Calloc(dimx, double);
    d = Calloc(n, double);
    y = Calloc(n, double);
    yf = Calloc(n, double);
    for (i = 0; i < n; i++) ord[i] = orde[i];
    for (i = 0; i < n; i++) ord2[i] = ordee[i];
    for (i = 0; i < dimx; i++) x[i] = xx[i];
}

void
VR_mds_unload()
{
    Free(ord); Free(ord2); Free(x); Free(d); Free(y); Free(yf);
}


static void
calc_dist(double *x)
{
    int   r1, r2, c, index;
    double tmp, tmp1;

    index = 0;
    for (r1 = 0; r1 < nr; r1++)
	for (r2 = r1 + 1; r2 < nr; r2++) {
	    tmp = 0.0;
	    for (c = 0; c < nc; c++) {
		tmp1 = x[r1 + c * nr] - x[r2 + c * nr];
		tmp += tmp1 * tmp1;
	    }
	    d[index++] = sqrt(tmp);
	}
    for (index = 0; index < n; index++)
	y[index] = d[ord[index]];
}

static double
fminfn(double *x)
{
    double ssq;
    Sint  do_derivatives = 0;

    calc_dist(x);
    VR_mds_fn(y, yf, &n, &ssq, ord2, x, &nr, &nc, 0, &do_derivatives);
    return (ssq);
}

static void
fmingr(double *x, double *der)
{
    double ssq;
    Sint  do_derivatives = 1;

    calc_dist(x);
    VR_mds_fn(y, yf, &n, &ssq, ord2, x, &nr, &nc, der, &do_derivatives);
}

void
VR_mds_dovm(double *val, Sint *maxit, Sint *trace, double *xx, double *tol)
{
    int   i;

    vmmin(dimx, x, val, (int) *maxit, (int) *trace, *tol);
    for (i = 0; i < dimx; i++)
	xx[i] = x[i];
}

/*
 *  Does isotonic regression.
 */

void
VR_mds_fn(double *y, double *yf, Sint *pn, double *pssq, Sint *pd,
	  double *x, Sint *pr, Sint *pncol, double *der,
	  Sint *do_derivatives)
{
    int   n = *pn, i, ip=0, known, u, s, r = *pr, ncol = *pncol, k=0;
    double tmp, ssq, *yc, slope, tstar, sstar;

    yc = Calloc((n + 1), double);
    yc[0] = 0.0;
    tmp = 0.0;
    for (i = 0; i < n; i++) {
	tmp += y[i];
	yc[i + 1] = tmp;
    }
    known = 0;
    do {
	slope = 1.0e+200;
	for (i = known + 1; i <= n; i++) {
	    tmp = (yc[i] - yc[known]) / (i - known);
	    if (tmp < slope) {
		slope = tmp;
		ip = i;
	    }
	}
	for (i = known; i < ip; i++)
	    yf[i] = (yc[ip] - yc[known]) / (ip - known);
    } while ((known = ip) < n);

    sstar = 0.0;
    tstar = 0.0;
    for (i = 0; i < n; i++) {
	tmp = y[i] - yf[i];
	sstar += tmp * tmp;
	tstar += y[i] * y[i];
    }
    ssq = 100 * sqrt(sstar / tstar);
    *pssq = ssq;
    Free(yc);
    if (!(*do_derivatives)) return;
    /* get derivatives */
    for (u = 0; u < r; u++) {
	for (i = 0; i < ncol; i++) {
	    tmp = 0.0;
	    for (s = 0; s < r; s++) {
		if (s > u)
		    k = r * u - u * (u + 1) / 2 + s - u;
		else if (s < u)
		    k = r * s - s * (s + 1) / 2 + u - s;
		if (s != u) {
		    k = pd[k - 1];
		    tmp += ((y[k] - yf[k]) / sstar
			    - y[k] / tstar) * (x[u + r * i] - x[s + r * i]) / y[k];
		}
	    }
	    der[u + i * r] = tmp * ssq;
	}
    }
}

/*  From here on, code borrowed from nnet library  */

static double *
vect(int n)
{
    double *v;

    v = (double *) Calloc(n, double);
    return v;
}

static void
free_vect(double *v)
{
    Free(v);
}

static double **
Lmatrix(int n)
{
    int   i;
    double **m;

    m = (double **) Calloc(n, double *);

    for (i = 0; i < n; i++) {
	m[i] = (double *) Calloc((i + 1), double);
    }
    return m;
}

static void free_Lmatrix(double **m, int n)
{
    int   i;

    for (i = n - 1; i >= 0; i--) Free(m[i]);
    Free(m);
}

typedef unsigned char Boolean;

#define false 0

#define stepredn	0.2
#define acctol		0.0001
#define reltest		10.0
#define abstol 		1.0e-2
#define REPORT		5


/*  BFGS variable-metric method, based on Pascal code
in J.C. Nash, `Compact Numerical Methods for Computers', 2nd edition,
converted by p2c then re-crafted by B.D. Ripley */

static void
vmmin(int n, double *b, double *Fmin, int maxit, int trace, double reltol)
{
    Boolean accpoint, enough;
    double *g, *t, *X, *c, **B;
    int     count, funcount, gradcount;
    double  f, gradproj;
    int     i, j, ilast, iter = 0;
    double  s, steplength;
    double  D1, D2;

    g = vect(n);
    t = vect(n);
    X = vect(n);
    c = vect(n);
    B = Lmatrix(n);
    f = fminfn(b);
    if (trace) {
	printf("initial  value %f \n", f);
	fflush(stdout);
    }
    {
	*Fmin = f;
	funcount = gradcount = 1;
	fmingr(b, g);
	iter++;
	ilast = gradcount;

	do {
	    if (ilast == gradcount) {
		for (i = 0; i < n; i++) {
		    for (j = 0; j < i; j++)
			B[i][j] = 0.0;
		    B[i][i] = 1.0;
		}
	    }

	    for (i = 0; i < n; i++) {
		X[i] = b[i];
		c[i] = g[i];
	    }
	    gradproj = 0.0;
	    for (i = 0; i < n; i++) {
		s = 0.0;
		for (j = 0; j <= i; j++) s -= B[i][j] * g[j];
		for (j = i + 1; j < n; j++) s -= B[j][i] * g[j];
		t[i] = s;
		gradproj += s * g[i];
	    }

	    if (gradproj < 0.0) {	/* search direction is downhill */
		steplength = 1.0;
		accpoint = false;
		do {
		    count = 0;
		    for (i = 0; i < n; i++) {
			b[i] = X[i] + steplength * t[i];
			if (reltest + X[i] == reltest + b[i])	/* no change */
			    count++;
		    }
		    if (count < n) {
			f = fminfn(b);
			funcount++;
			accpoint = (f <= *Fmin + gradproj * steplength * acctol);

			if (!accpoint) {
			    steplength *= stepredn;
			}
		    }
		} while (!(count == n || accpoint));
		enough = (f > abstol) && (f < (1.0 - reltol) * (*Fmin));
		/* stop if value if small or if relative change is low */
		if (!enough) count = n;
		if (count < n) {	/* making progress */
		    *Fmin = f;
		    fmingr(b, g);
		    gradcount++;
		    iter++;
		    D1 = 0.0;
		    for (i = 0; i < n; i++) {
			t[i] = steplength * t[i];
			c[i] = g[i] - c[i];
			D1 += t[i] * c[i];
		    }
		    if (D1 > 0) {
			D2 = 0.0;
			for (i = 0; i < n; i++) {
			    s = 0.0;
			    for (j = 0; j <= i; j++)
				s += B[i][j] * c[j];
			    for (j = i + 1; j < n; j++)
				s += B[j][i] * c[j];
			    X[i] = s;
			    D2 += s * c[i];
			}
			D2 = 1.0 + D2 / D1;
			for (i = 0; i < n; i++) {
			    for (j = 0; j <= i; j++)
				B[i][j] += (D2 * t[i] * t[j] - X[i] * t[j] - t[i] * X[j]) / D1;
			}
		    } else {	/* D1 < 0 */
			ilast = gradcount;
		    }
		} else {	/* no progress */
		    if (ilast < gradcount) {
			count = 0;
			ilast = gradcount;
		    }
		}
	    } else {		/* uphill search */
		count = 0;
		if (ilast == gradcount) count = n;
		else ilast = gradcount;
		/* Resets unless has just been reset */
	    }
	    if (iter % REPORT == 0 && trace) {
		printf("iter%4d value %f\n", iter, f);
		fflush(stdout);
	    } if (iter >= maxit)
		break;
	    if (gradcount - ilast > 2 * n)
		ilast = gradcount;	/* periodic restart */
	} while (count != n || ilast != gradcount);
    }
    if (trace) {
	printf("final  value %f \n", *Fmin);
	if (iter < maxit)
	    printf("converged\n");
	else
	    printf("stopped after %i iterations\n", iter);
    }
    free_vect(g);
    free_vect(t);
    free_vect(X);
    free_vect(c);
    free_Lmatrix(B, n);
}

/* -----------------------------------------------------------------
 *  Former ucv.c
 */

#if !defined(PI)		/* it is currently defined in S_tokens.h */
#  define PI 3.14159265
#endif
#define DELMAX 1000
/* Avoid slow and possibly error-producing underflows by cutting off at
   plus/minus sqrt(DELMAX) std deviations */
/* Formulae (6.67) and (6.69) of Scott (1992), the latter corrected. */

void
VR_ucv_bin(Sint *n, Sint *nb, Sfloat *d, Sint *x, Sfloat *h, Sfloat *u)
{
    int   i, nn = *n, nbin = *nb;
    Sfloat delta, hh = (*h) / 4, sum, term;

    sum = 0.0;
    for (i = 0; i < nbin; i++) {
	delta = i * (*d) / hh;
	delta *= delta;
	if (delta >= DELMAX) break;
	term = exp(-delta / 4) - sqrt(8.0) * exp(-delta / 2);
	sum += term * x[i];
    }
    *u = 1 / (2 * nn * hh * sqrt(PI)) + sum / (nn * nn * hh * sqrt(PI));
}

void
VR_bcv_bin(Sint *n, Sint *nb, Sfloat *d, Sint *x, Sfloat *h, Sfloat *u)
{
    int   i, nn = *n, nbin = *nb;
    Sfloat delta, hh = (*h) / 4, sum, term;

    sum = 0.0;
    for (i = 0; i < nbin; i++) {
	delta = i * (*d) / hh;
	delta *= delta;
	if (delta >= DELMAX) break;
	term = exp(-delta / 4) * (delta * delta - 12 * delta + 12);
	sum += term * x[i];
    }
    *u = 1 / (2 * nn * hh * sqrt(PI)) + sum / (64 * nn * nn * hh * sqrt(PI));
}


void
VR_phi4_bin(Sint *n, Sint *nb, Sfloat *d, Sint *x, Sfloat *h, Sfloat *u)
{
    int   i, nn = *n, nbin = *nb;
    Sfloat delta, sum, term;

    sum = 0.0;
    for (i = 0; i < nbin; i++) {
	delta = i * (*d) / (*h);
	delta *= delta;
	if (delta >= DELMAX) break;
	term = exp(-delta / 2) * (delta * delta - 6 * delta + 3);
	sum += term * x[i];
    }
    sum = 2 * sum + nn * 3;	/* add in diagonal */
    *u = sum / (nn * (nn - 1) * pow(*h, 5.0) * sqrt(2 * PI));
}

void
VR_phi6_bin(Sint *n, Sint *nb, Sfloat *d, Sint *x, Sfloat *h, Sfloat *u)
{
    int   i, nn = *n, nbin = *nb;
    Sfloat delta, sum, term;

    sum = 0.0;
    for (i = 0; i < nbin; i++) {
	delta = i * (*d) / (*h);
	delta *= delta;
	if (delta >= DELMAX) break;
	term = exp(-delta / 2) *
	    (delta * delta * delta - 15 * delta * delta + 45 * delta - 15);
	sum += term * x[i];
    }
    sum = 2 * sum - 15 * nn;	/* add in diagonal */
    *u = sum / (nn * (nn - 1) * pow(*h, 7.0) * sqrt(2 * PI));
}

void
VR_den_bin(Sint *n, Sint *nb, Sfloat *d, Sfloat *x, Sint *cnt)
{
    int   i, j, ii, jj, iij, nn = *n;
    Sfloat xmin, xmax, rang, dd;

    for (i = 0; i < *nb; i++) cnt[i] = 0;
    xmin = xmax = x[0];
    for (i = 1; i < nn; i++) {
	xmin = min(xmin, x[i]);
	xmax = max(xmax, x[i]);
    }
    rang = (xmax - xmin) * 1.01;
    *d = dd = rang / (*nb);
    for (i = 1; i < nn; i++) {
	ii = (int) (x[i] / dd);
	for (j = 0; j < i; j++) {
	    jj = (int) (x[j] / dd);
	    iij = abs9((ii - jj));
	    cnt[iij]++;
	}
    }
}

/* find maximum column: designed for probabilities. Uses
   reservoir sampling to break ties at random */

#define RELTOL 1e-5

void
VR_max_col(double *matrix, int *pnr, int *nc, int *maxes)
{
    int   r, c, m, nr = *pnr, ntie;
    double a, b, tol, large;
    int isna;

    S_EVALUATOR
    RANDIN;
    for (r = 0; r < nr; r++) {
	/* first check row for any NAs and find the largest entry */
	large = 0.0;
	isna = 0;
	for (c = 0; c < *nc; c++) {
	    a = matrix[r + c * nr];
#if( SPLUS_VERSION >= 6000 )
	    if (is_na(&a, S_MODE_DOUBLE)) { isna = 1; break; }
#else
	    if (is_na(&a, DOUBLE)) { isna = 1; break; }
#endif
	    large = max(large, fabs(a));
	}
	if (isna) { 
#if( SPLUS_VERSION >= 6000 )
	    na_set(maxes + r, S_MODE_INT);
#else
	    na_set(maxes + r, INT);
#endif
	    continue; 
	}
	tol = RELTOL * large;

	m = 0;
	ntie = 1;
	a = matrix[r];
	for (c = 1; c < *nc; c++) {
	    b = matrix[r + c * nr];
	    if (b >= a + tol) {
		ntie = 1;
		a = b;
		m = c;
	    } else if (b >= a - tol) {
		ntie++;
		if (ntie * UNIF < 1.0) m = c;
	    }
	}
	maxes[r] = m + 1;
    }
    RANDOUT;
}

/* -----------------------------------------------------------------
 *  Former lqs.c
 */


/* corrected from R. Sedgewick `Algorithms in C' */
static void
shellsort(double *a, int N)
{
    int   i, j, h;
    double v;

    for (h = 1; h <= N / 9; h = 3 * h + 1);
    for (; h > 0; h /= 3)
	for (i = h; i < N; i++) {
	    v = a[i];
	    j = i;
	    while (j >= h && a[j - h] > v) {
		a[j] = a[j - h];
		j -= h;
	    }
	    a[j] = v;
	}
}

/*
   Partial sort so that a[k] is in the correct place, smaller to left,
   larger to right
 */
static void
psort(double *a, int N, int k)
{
    int   L, R, i, j;
    double v, tmp;

    for (L = 0, R = N - 1; L < R;) {
	v = a[k];
	for (i = L, j = R; i <= j;) {
	    while (a[i] < v) i++;
	    while (v < a[j]) j--;
	    if (i <= j) {
		tmp = a[i];
		a[i++] = a[j];
		a[j--] = tmp;
	    }
	}
	if (j < k) L = i;
	if (k < i) R = j;
    }
    if (j < k) L = i;
    if (k < i) R = j;
}

/* find qr decomposition, basis of qr.default() */
void F77_NAME(dqr)(double *qr, Sint *dx, Sint *pivot, double *qraux,
		   double *tol, double *work, Sint *rank);

/* solve for coefficients */
void F77_NAME(dqrsl)(double *qr, Sint *ldx, Sint *n, Sint *rank,
		     double *qraux,
		     double *y1, double *d1, double *y2, double *coef,
		     double *d3, double *d4, Sint *job, Sint *info);


static double *coef, *qraux, *work, *res, *yr, *xr, *means, *d2, *d2copy;
static Sint *pivot, *which, *which2, *bestone;
static int *ind;

/*
   Sampling k from 0:n-1 without replacement.
 */
static void
sample_noreplace(Sint *x, int n, int k)
{
    int   i, j, nn = n;

    for (i = 0; i < n; i++) ind[i] = i;
    for (i = 0; i < k; i++) {
	j = (int) (nn * UNIF);
	x[i] = ind[j];
	ind[j] = ind[--nn];
    }
}

/*
   Find all subsets of size k in order: this gets a new one each call
 */
static void
next_set(Sint *x, int n, int k)
{
    int   i, j, tmp;

    j = k - 1;
    tmp = x[j]++;
    while (j > 0 && x[j] >= n - (k - 1 - j)) tmp = ++x[--j];
    for (i = j + 1; i < k; i++) x[i] = ++tmp;
}


static void
lqs_setup(Sint *n, Sint *p, Sint *ps)
{
    coef = Salloc(*p, double);
    qraux = Salloc(*p, double);
    work = Salloc(2 * (*p), double);
    res = Salloc(*n, double);
    yr = Salloc(*n, double);
    xr = Salloc((*n) * (*p), double);
    pivot = Salloc(*p, Sint);
    ind = Salloc(*n, int);
    which = Salloc(*ps, Sint);
    bestone = Salloc(*ps, Sint);
}
/*
   Adjust the constant for an LMS fit. This is the midpoint of the
   qn contiguous observations of shortest length.
 */
static double
lmsadj(double *x, int n, int qn, double *ssbest)
{
    int   i, k = qn - 1;
    double len, best, adj;

    best = x[k] - x[0];
    adj = 0.5 * (x[k] + x[0]);
    for (i = 1; i < n - k; i++) {
	len = x[i + k] - x[i];
	if (len < best) {
	    best = len;
	    adj = 0.5 * (x[i + k] + x[i]);
	}
    }
    *ssbest = 0.25 * best * best;
    return (adj);
}

/*
   Adjust the constant for an LTS fit. This is the mean of the
   qn contiguous observations of smallest variance.
 */
static double
ltsadj(double *x, int n, int qn, double *ssbest)
{
    int   i, k = qn - 1;
    double ss, best, m1, m2, adj;

    /* printf("qn = %d\n", qn); */
    m1 = m2 = 0.0;
    for (i = 0; i < qn; i++) {
	m1 += x[i];
	m2 += x[i] * x[i];
    }
    adj = m1 / qn;
    best = m2 - m1 * m1 / qn;

    for (i = 1; i < n - k; i++) {
	m1 += x[i + k] - x[i - 1];
	m2 += x[i + k] * x[i + k] - x[i - 1] * x[i - 1];
	ss = m2 - m1 * m1 / qn;
	if (ss < best) {
	    best = ss;
	    adj = m1 / qn;
	}
    }
    *ssbest = best;
    return (adj);
}

/* the chi function for the S estimator: the integral of biweight */
static double
chi(double x, double a)
{
    x /= a; x *= x;
    if(x > 1) return(1.0);
    else return(x*(3 + x*(-3 + x)));
}

/*
   For lots of subsets of size *nwhich, compute the exact fit to those
   data points and the residuals from all the data points.
 */
void
lqs_fitlots(double *x, double *y, Sint *n, Sint *p, Sint *qn,
	    Sint *lts, Sint *adj, Sint *sample, Sint *nwhich,
	    Sint *ntrials, double *crit, Sint *sing, Sint *bestone,
	    double *bestcoef, double *pk0, double *beta)
{
    Sint  i, iter, j, k, nn = *n, nnew = *nwhich, this, trial;
    Sint  dx[2], rank, info, n100 = 100;
    double a = 0.0, tol = 1.0e-7, sum, thiscrit, best = 1.e20, target, old,
          new, dummy, k0 = *pk0;
    int firsttrial = 1;

    S_EVALUATOR
	lqs_setup(n, p, nwhich);

    *sing = 0;
    target = (nn - *p) * (*beta);

    if(!(*sample)) {
	for(i = 0; i < nnew; i++) which[i] = i;
    } else RANDIN;

    for(trial = 0; trial < *ntrials; trial++) {
	if(!(*sample)) {
	    if(trial > 0) next_set(which, nn, nnew);
	} else
	    sample_noreplace(which, nn, nnew);

	for(j = 0; j < nnew; j++) {
	    this = which[j];
	    yr[j] = y[this];
	    for (k = 0; k < *p; k++)
		xr[j + nnew * k] = x[this + nn * k];
	}

	/* compute fit, find residuals */
	dx[0] = nnew;
	dx[1] = *p;
	for (i = 0; i < *p; i++)
	    pivot[i] = i + 1;
	rank = *p;
	F77_CALL(dqr) (xr, dx, pivot, qraux, &tol, work, &rank);
	if (rank < *p) {
	    (*sing)++;
	    continue;
	}
	F77_CALL(dqrsl) (xr, &nnew, &nnew, &rank, qraux, yr, &dummy, yr,
			 coef, &dummy, &dummy, &n100, &info);

	for (i = 0; i < nn; i++) {
	    sum = y[i];
	    for (j = 0; j < rank; j++)
		sum -= coef[j] * x[i + nn * j];
	    res[i] = sum;
	}

	if (*lts < 2) {		/* lqs or lts estimation */
	    /* find the constant subtracted from the residuals that minimizes
	       the criterion. As this is a univariate problem, has an exact
	       solution.  */
	    if (*adj) {
		shellsort(res, nn);
		if (*lts)
		    a = ltsadj(res, nn, *qn, &thiscrit);
		else
		    a = lmsadj(res, nn, *qn, &thiscrit);
	    } else {
		for(i = 0; i < nn; i++) {
		    sum = res[i] - a;
		    res[i] = sum*sum;
		}
		psort(res, nn, *qn - 1);	/* partial sort */
		if (!(*lts))
		    thiscrit = res[*qn - 1];
		else {
		    sum = 0.0;
		    for (i = 0; i < *qn; i++)
			sum += res[i];
		    thiscrit = sum;
		}
	    }
	} else {		/* S estimation */
	    if (firsttrial) {
		for (i = 0; i < nn; i++)
		    res[i] = fabs(res[i]);
		psort(res, nn, nn / 2);
		old = res[nn / 2] / 0.6745;	/* MAD provides the initial
						   scale */
		firsttrial = 0;
	    } else {
		/* only find optimal scale if it will be better than existing
		   best solution */
		sum = 0.0;
		for (i = 0; i < nn; i++)
		    sum += chi(res[i], k0 * best);
		if (sum > target)
		    continue;
		old = best;
	    }			/* now solve for scale S by re-substitution */
	    for (iter = 0; iter < 30; iter++) {
		/* printf("iter %d, s = %f sum = %f %f\n", iter, old, sum,
		   target); */
		sum = 0.0;
		for (i = 0; i < nn; i++)
		    sum += chi(res[i], k0 * old);
		new = sqrt(sum / target) * old;
		if (fabs(sum / target - 1.) < 1e-4)
		    break;
		old = new;
	    }
	    thiscrit = new;
	}

	if (thiscrit < best) {	/* first trial might be singular, so use
				   fence */
	    sum = 0.0;
	    for (i = 0; i < nn; i++)
		sum += chi(res[i], k0 * best);
	    best = thiscrit;
	    /* printf("trial %d, best = %f sum = %f %f\n", trial, best, sum,
	       target); */
	    for (i = 0; i < nnew; i++)
		bestone[i] = which[i] + 1;
	    for (i = 0; i < *p; i++)
		bestcoef[i] = coef[i];
	    bestcoef[0] += a;
	}
    }
    *crit = (best < 0.0) ? 0.0 : best;
    if(*sample) RANDOUT;
}


static void
mve_setup(Sint * n, Sint * p, Sint * ps)
{
    xr = Salloc((*ps) * (*p), double);
    qraux = Salloc(*p, double);
    pivot = Salloc(*p, Sint);
    work = Salloc(2 * (*p), double);
    d2 = Salloc(*n, double);
    d2copy = Salloc(*n, double);
    means = Salloc((*p), double);
    ind = Salloc(*n, int);
    which = Salloc(*ps, Sint);
    which2 = Salloc(*ps, Sint);
    bestone = Salloc(*n, Sint);
}


/* find the squared Mahalanobis distance to x via QR decomposition in xr. */
static double
mah(double *xr, int nnew, int p, double *x)
{
    int   i, j;
    double s, ss = 0.0;

    for (j = 0; j < p; j++) {
	s = x[j];
	if (j > 0)
	    for (i = 0; i < j; i++)
		s -= work[i] * xr[i + nnew * j];
	work[j] = s / xr[j + nnew * j];
	ss += work[j] * work[j];
    }
    return (ss * (nnew - 1));
}

/*
   Compute the squared Mahalanobis distances, in d2, to all points in x
   from the mean of the subset in which using the covariance of that
   subset.
*/
static int
do_one(double *x, Sint *which, int n, Sint nnew, Sint p,
       double *det, double *d2)
{
    int   i, j, k;
    Sint  rank, dx[2];
    double sum, tol = 1.0e-7;

    for (j = 0; j < nnew; j++)
	for (k = 0; k < p; k++)
	    xr[j + nnew * k] = x[which[j] + n * k];
    for (k = 0; k < p; k++) {
	sum = 0.0;
	for (j = 0; j < nnew; j++)
	    sum += xr[j + nnew * k];
	sum /= nnew;
	means[k] = sum;
	for (j = 0; j < nnew; j++)
	    xr[j + nnew * k] -= sum;
    }

    dx[0] = nnew;
    dx[1] = p;
    rank = p;
    for (i = 0; i < p; i++)
	pivot[i] = i + 1;
    F77_CALL(dqr) (xr, dx, pivot, qraux, &tol, work, &rank);
    if (rank < p)
	return (1);

    sum = 0.0;
    for (k = 0; k < p; k++)
	sum += log(fabs(xr[k + nnew * k]));
    *det = sum;

    /* now solve R^T b = (x[i, ] - means) and find squared length of b */
    for (i = 0; i < n; i++) {
	for (j = 0; j < p; j++)
	    qraux[j] = x[i + n * j] - means[j];
	d2[i] = mah(xr, nnew, p, qraux);
    }
    return (0);
}


void
mve_fitlots(double *x, Sint *n, Sint *p, Sint *qn, Sint *mcd,
	    Sint *sample, Sint *nwhich, Sint *ntrials,
	    double *crit, Sint *sing, Sint *bestone)
{
    int   i, iter, j, nn = *n, quan = *qn, trial, this_sing;
    Sint  nnew = *nwhich;
    double det, best = 1.e20, thiscrit, lim;

    S_EVALUATOR
    if (*mcd != 1)
	mve_setup(n, p, nwhich);
    else
	mve_setup(n, p, n);	/* could get ties */

    *sing = 0;
    if (!*sample) {
	for (i = 0; i < nnew; i++)
	    which[i] = i;
    } else
	RANDIN;

    thiscrit = 0.0;		/* -Wall */

    for (trial = 0; trial < *ntrials; trial++) {

	if (!(*sample)) {
	    if (trial > 0)
		next_set(which, nn, nnew);
	} else
	    sample_noreplace(which, nn, nnew);

	/* for(i = 0; i < nnew; i++) printf("%d ", which[i]); printf("\n");
	   fflush(stdout); */


	/* Find the mean and covariance matrix of the sample. Check if
	   singular. Compute Mahalanobis distances of all points to the means
	   using this covariance matrix V, and find quantile largest. Volume
	   is then monotone in determinant(V * dist2). */

	this_sing = do_one(x, which, nn, nnew, *p, &det, d2);
	if (this_sing) {
	    (*sing)++;
	    continue;
	}
	/* for(i = 0; i < nnew; i++) printf(" %d", which[i]); printf("\n"); */

	for (i = 0; i < nn; i++)
	    d2copy[i] = d2[i];
	psort(d2copy, nn, quan - 1);
	lim = d2copy[*qn - 1];
	if (!*mcd)
	    thiscrit = (*p) * log(lim) + 2 * det;
	else {
	    for (iter = 0; iter < 4; iter++) {
		/* for(i = 0; i < nn; i++) printf(" %f", d2[i]);
		   printf("\n"); */
		if (iter > 0) {
		    for (i = 0; i < nn; i++)
			d2copy[i] = d2[i];
		    psort(d2copy, nn, quan - 1);
		    lim = d2copy[*qn - 1];
		}
		j = 0;
		for (i = 0; i < nn; i++)
		    if (d2[i] <= lim)
			which2[j++] = i;
		/* note: we take all points that meet this limit: there could
		   be more than quan. */
		(void) do_one(x, which2, nn, quan, *p, &det, d2);
		if (iter > 0 && 2 * det >= 0.999 * thiscrit)
		    break;
		thiscrit = 2 * det;
		/* printf("iter %d %f", iter, thiscrit); for(i = 0; i < quan;
		   i++) printf(" %d", which2[i]); printf("\n");
		   fflush(stdout); */
	    }

	}
	/* printf("this %f\n", thiscrit); */


	if (thiscrit < best) {	/* warning: first might be singular */
	    best = thiscrit;
	    for (i = 0; i < nn; i++)
		bestone[i] = (d2[i] <= lim);
	}
    }
    *crit = best;
    if(*sample) RANDOUT;
}
