/*  survnnet/survnnet.c by R. M. Ripley 
 *  based on nnet/nnet.c by B. D. Ripley  Copyright (C) 1992-8
 *
 * weights are stored in order of their destination unit.
 * the array Conn gives the source unit for the weight (0 = bias unit)
 * the array Nconn gives the number of first weight connecting to each unit,
 * so the weights connecting to unit i are Nconn[i] ... Nconn[i+1] - 1.
 *
 */

#include <R.h>
#include <Rmath.h>
#include <stdio.h>


static double safepow(double x, double y);
static double *dvect(int n);
static int *ivect(int n);
static float *vect(int n);
static float **matrix(int nrh, int nch);
static double **dmatrix(int nr, int nc);
static double ***d3matrix(int nr, int nc, int nd);
static double ****d4matrix(int nr, int nc, int nd, int ne);
static double *****d5matrix(int nr, int nc, int nd, int ne, int nf);
static void free_matrix(float **m, int nrh, int nch);
static void free_dmatrix(double **m, int nr, int nc);
static void free_d3matrix(double ***m, int nr, int nc, int nd);
static void free_vect(float *v);
static void free_ivect(int *v);
static void free_dvect(double *v);
static void vmmin(int n, double *b, double *Fmin, int maxit, int trace, int *mask, float abstol);
static void fpass(float *input, float *goal, float wx, float stime, int subj, int all);
static void Build_Net(int ninputs, int nhidden, int noutputs);
static void ModelHess(float *input, float *goal, float wx, float stime, int subj);
static void bpass2(float *goal, float wx, float stime);
static void phHessian(double *inwts, double *Hess);
static void calcOuts(float *input, int subj, float stime);
static double model5(double y, double t, float stime, int subj);

static int      phflag; /* values 0 for not, 1 for phnet, 2 for phtnet*/


/* #define N 100			Max number of units allowed in net */
/* #define NWTS 1000 */

static int Epoch;
static double* Decay;
static double TotalError;
static double Alpha;
static double dAlpha;

static int Nintervals;
static float varWt;

static int Nunits;
static int Ninputs;
static int Nhidden;
static int FirstHidden;
static int FirstOutput;
static int Noutputs;
static int NSunits;		/* number of sigmoid units */
static int Nweights;
static int Entropy;
static int Linout;
static int Softmax;
static int Censored;
static int Model;

static double* Outputs;
static double ***Outputs5;
static double* ErrorSums;
static double* Errors;
static int* Nconn;
static int* Conn;
static int	*Dest;
static double* wts;
static double* Slopes;
static double* Probs;
static double	*ptexp, **ptexpy, ***ptexpwyx, ***ptexpyy, ***ptexpyx, ****ptexpwyyx,
****ptexpwYxx, *****ptexpwyxwyx, **ptexpx, ***ptexpxx, ****ptexpxwyx, ***ptexpxy;
static double	psexp, *psexpy, **psexpwyx, **psexpyy, **psexpyx, ***psexpwyyx,
***psexpwYxx, ****psexpwyxwyx, *psexpx, **psexpxx, ***psexpxwyx, **psexpxy;
static double   **tmpij, *tmp, *tmpx;

static int NTrain;
static int NTest;
static int	NPred;
static int      Nzone;
static float **TrainIn;
static float **TrainOut;
static float **TestIn;
static float **TestOut;
static float *Weights;
static float *TrainStime;
static float	**PredIn;
static int	*Status;
static int      *Zone;
static float    *Uzone;
static double	***phTrainOut;
static float	**PredOut;
static double	***HiddenOut;
static float *trainInt;

static float* toutputs;

static void
errmsg(char *string)
{
  PROBLEM "%s\n", string RECOVER(NULL_ENTRY);
}

void
set_survnet(int *n, int *nconn, int *conn,
	   double *decay, int *nsunits, int *entropy,
	   int *softmax, int *censored, int *model, float *alpha,
	    int *intervals, float *varwt)
{
  int     i;

  phflag=0;
  
  Build_Net((int) n[0], (int) n[1], (int) n[2]);
  for (i = 0; i <= Nunits; i++) Nconn[i] = nconn[i];
  Nweights = Nconn[Nunits];
  Conn = Calloc(Nweights, int);
  wts = Calloc(Nweights, double);
  Slopes = Calloc(Nweights, double);
  Probs = Calloc(Nweights, double);
  Decay = Calloc(Nweights, double);
  for (i = 0; i < Nweights; i++) Conn[i] = conn[i];
  Epoch = 0;
  for (i = 0; i < Nweights; i++) Decay[i] = decay[i];
  TotalError = 0.0;
  NSunits = *nsunits;
  Entropy = *entropy;
  Linout = (NSunits < Nunits);
  Softmax = *softmax;
  Censored = *censored;
  Model =*model;
  Alpha = *alpha;
  Nintervals=*intervals;
  varWt=*varwt;
}
void 
unset_survnet()
{
  Free(Conn); Free(wts); Free(Slopes); Free(Probs); Free(Decay);
  Free(Nconn); Free(Outputs); Free(ErrorSums); Free(Errors); Free(toutputs);
}

void
set_survtrain(int *ntr, float *train, float *weights, float *stime)
{
  int     i, j;

  NTrain = *ntr;
  TrainIn = matrix(NTrain, Ninputs);
  TrainOut = matrix(NTrain, Noutputs);
  if (Model==5) 
  {
      Outputs5=d3matrix(NTrain, Nunits,Nintervals);
      for (i= 0; i< Nintervals; i++)
	  for (j=0;j<NTrain;j++)
	      Outputs5[j][0][i]=1.0;
  }
  
  Weights = vect(NTrain);
  TrainStime = vect(NTrain);
  trainInt = vect(NTrain);
  for (j = 0; j < Ninputs; j++)
    for (i = 0; i < NTrain; i++) TrainIn[i][j] = *train++;
  for (j = 0; j < Noutputs; j++)
    for (i = 0; i < NTrain; i++) TrainOut[i][j] = *train++;
  for (i = 0; i < NTrain; i++) Weights[i] = *weights++;
  for (i = 0; i < NTrain; i++) TrainStime[i] = *stime++;
}

void 
unset_survtrain()
{
  free_matrix(TrainIn, NTrain, Ninputs);
  free_matrix(TrainOut, NTrain, Noutputs);
  if (Model==5)free_d3matrix(Outputs5, NTrain,Nunits,Nintervals);
  free_vect(Weights);
  free_vect(TrainStime);
  free_vect(trainInt);
}

void
survnntest(int *ntest, float *test, float *result, double *inwts, int *llike, int *allhaz, float *likeres, double *hazres)
{
  int     i, j, loglike, allHaz;
  double *intc;
  float stime;

  for (i = 0; i < Nweights; i++) wts[i] = inwts[i];
  NTest = *ntest;
  if (Nweights == 0) errmsg("No model set");
  TestIn = matrix(NTest, Ninputs);
  TestOut = matrix(NTest, Noutputs);
  intc = dvect(NTest);
  
  allHaz=*allhaz;  
  loglike=*llike;
  if (loglike||allHaz) 
    {
      Outputs5=d3matrix(NTest, Nunits,Nintervals);
      for (i= 0; i< Nintervals; i++)
	for (j=0;j<NTest;j++)
	  Outputs5[j][0][i]=1.0;
    }
  
 for (i = 0; i < Ninputs; i++)
    for (j = 0; j < NTest; j++)
      TestIn[j][i] = *test++;

  for (i = 0; i < Noutputs; i++) toutputs[i] = 0.5;
  for (j = 0; j < NTest; j++) {
      if (loglike||allHaz) 
      {
	  stime=TestIn[j][Ninputs-1];
	  calcOuts(TestIn[j],j,stime);
	  intc[j]= model5(1.0,0.0,stime,j);
      }
    fpass(TestIn[j], toutputs, 1.0, 1.0, 1, 0);
    if (Softmax)
      for (i = 0; i < Noutputs; i++) TestOut[j][i] = Probs[FirstOutput + i];
    else
      for (i = 0; i < Noutputs; i++) TestOut[j][i] = Outputs[FirstOutput + i];
  }
  for (i = 0; i < Noutputs; i++)
    for (j = 0; j < NTest; j++) *result++ = TestOut[j][i];
   if (loglike)
      for (j =0;j<NTest;j++)  *likeres++ = intc[j];
  if (allHaz)
      for (j=0;j<Nintervals;j++)
	  for (i=0;i<NTest;i++)
	      *hazres++= Outputs5[i][FirstOutput][j];
  
  free_dvect(intc);
  free_matrix(TestIn, NTest, Ninputs);
  free_matrix(TestOut, NTest, Noutputs);
  if (loglike||allHaz) 
      free_d3matrix(Outputs5,NTest, Nunits,Nintervals);
}

void set_phtnet(int *n, int *nconn, int *conn, int *nsunits, float *decay)
{
   int i;
   Build_Net((int)n[0], (int)n[1], (int)n[2]);
   Nhidden= (int)n[1];
   for (i = 0; i <= Nunits; i++) Nconn[i] = nconn[i];
   Nweights = Nconn[Nunits];
   Conn = Calloc(Nweights, int);
   Decay = Calloc(Nweights, double);
   Dest = Calloc(Nweights, int);
   wts = Calloc(Nweights, double);
   for (i = 0; i < Nweights; i++) Conn[i] = conn[i];
   NSunits = *nsunits;
   Linout = (NSunits < Nunits);
   for (i = 0; i < Nweights; i++) Decay[i] = decay[i];
}
void 
unset_phtnet()
{
    Free(Conn); Free(wts);  Free(Decay); Free(Dest); 
    Free(Nconn); Free(Outputs); Free(ErrorSums); Free(Errors); Free(toutputs);
}

void set_phttrain(int *ntr, float *train, int *status, int *nzone, int *zone, float *uzone)
{
    int i,j;
    phflag = 2;
    NTrain = *ntr;
    Nzone = *nzone;
    TrainIn = matrix(NTrain,Ninputs);
    Status = ivect(NTrain);
    Zone = ivect(NTrain);    
    phTrainOut = d3matrix(NTrain,Noutputs+1,Nzone);
    HiddenOut = d3matrix(NTrain,Nhidden, Nzone);
    for(j=0; j < Ninputs; j++)
	for(i=0; i < NTrain; i++) TrainIn[i][j] = *train++; 
    for (i=0; i < NTrain; i++)
    {
	Zone[i] = *zone++;
	Status[i] = *status++;
    }
    Uzone = vect(Nzone);    
    for (i = 0; i < Nzone; i++)
	Uzone[i] = *uzone++;
    ptexpy = dmatrix(Nhidden, Nzone);
    ptexpx = dmatrix(Ninputs,Nzone);
    ptexpwyx = d3matrix(Ninputs+1,Nhidden,Nzone);
    tmpij = dmatrix(Ninputs+1,Nhidden);
    ptexp = dvect(Nzone);
    tmpx = dvect(Ninputs);
    if (Nhidden> 0) tmp = dvect(Nhidden);
}
void unset_phttrain(void)
{
  free_matrix(TrainIn, NTrain, Ninputs);
  free_d3matrix(phTrainOut, NTrain, Noutputs+1,Nzone);
  free_ivect(Status);
  free_ivect(Zone);
  free_vect(Uzone);
  free_d3matrix(HiddenOut,NTrain,Nhidden,Nzone);
  free_dmatrix(ptexpx,Ninputs,Nzone);
  free_d3matrix(ptexpwyx, Ninputs+1,Nhidden,Nzone);
  free_dmatrix(tmpij,Ninputs+1,Nhidden);
  if (Nhidden>0) free_dvect(tmp);
  free_dvect(tmpx);
  free_dvect(ptexp);
  free_dmatrix(ptexpy,Nhidden,Nzone);
}

void set_phtrain(int *ntr, float *train, int *status)
{
   int i,j;
    phflag = 1;
   NTrain = *ntr;
   TrainIn = matrix(NTrain,Ninputs);
   Status = ivect(NTrain);
   phTrainOut = d3matrix(NTrain,Noutputs+1,1);
   HiddenOut = d3matrix(NTrain,Nhidden,1);
   for(j=0; j < Ninputs; j++)
       for(i=0; i < NTrain; i++) TrainIn[i][j] = *train++; 
   for (i=0; i < NTrain; i++)
       Status[i] = *status++;
   if (Nhidden > 0) psexpy = dvect(Nhidden);
   psexpx = dvect(Ninputs);
   psexpwyx = dmatrix(Ninputs+1,Nhidden);
   tmpij = dmatrix(Ninputs+1,Nhidden);
   tmpx = dvect(Ninputs);
   if (Nhidden> 0) tmp = dvect(Nhidden);
}
void unset_phtrain()
{
  free_matrix(TrainIn, NTrain, Ninputs);
  free_d3matrix(phTrainOut, NTrain, Noutputs+1,1);
  free_ivect(Status);
  free_d3matrix(HiddenOut,NTrain,Nhidden,1);
  free_dvect(psexpx);
  free_dmatrix(psexpwyx, Ninputs+1,Nhidden);
  free_dmatrix(tmpij,Ninputs+1,Nhidden);
  if (Nhidden> 0) free_dvect(psexpy);
  if (Nhidden> 0) free_dvect(tmp);
  free_dvect(tmpx);
}
static double
sigmoid(double sum)
{
  if (sum < -100.0) return (0.0);
  else if (sum > 100.0) return (1.0);
  else return (1.0 / (1.0 + exp(-sum)));
}


void
pred_phtnnet(int *npred, float *x, double *pwts, float *pred, int *nzone, 
	     float *uzone)
{
    int             i, j, k, l;
    double 	    sum;
    for (i = 0; i < Nweights; i++)
	wts[i] = pwts[i];
    NPred = *npred;
    Nzone = *nzone;
    PredIn = matrix(NPred, Ninputs);
    PredOut = matrix(NPred, Nzone);
    Uzone =  vect(Nzone);
    for (i = 0; i < Nzone; i++)
	Uzone[i] = *uzone++;
    
    for (j = 0; j < Ninputs; j++)
	for (i = 0; i < NPred; i++)
	    PredIn[i][j] = *x++;
    for (k = 0; k < NPred; k++)
    {
	for (l = 0; l < Nzone; l++)
	{
	    PredIn[k][Ninputs-1]=Uzone[l];
	    for (i = 0; i < Ninputs; i++)
		Outputs[i + 1] = PredIn[k][i];
	    
	    for (j = FirstHidden; j < Nunits; j++)
	    {
		sum = 0.0;
		for (i = Nconn[j]; i < Nconn[j + 1]; i++)
		    sum += Outputs[Conn[i]] * wts[i];
		if (j < NSunits)
		    sum = sigmoid(sum);
		Outputs[j] = sum;
		if (j >= FirstOutput)
		    PredOut[k][l] = sum;
	    }
	}
	
    }
    for (l = 0; l < Nzone; l++)
	for (k = 0; k < NPred; k++)
	    *pred++ = PredOut[k][l];
    free_matrix(PredIn, NPred, Ninputs);
    free_matrix(PredOut, NPred, Nzone);
    free_vect(Uzone);
}
void
pred_phnnet(int *npred, float *x, double *pwts, float *pred)
{
    int             i, j, k;
    double 	    sum;
    for (i = 0; i < Nweights; i++)
	wts[i] = pwts[i];
    NPred = *npred;
    PredIn = matrix(NPred, Ninputs);
    PredOut = matrix(NPred, Noutputs);
    for (j = 0; j < Ninputs; j++)
	for (i = 0; i < NPred; i++)
	    PredIn[i][j] = *x++;
    for (k = 0; k < NPred; k++)
    {
	for (i = 0; i < Ninputs; i++)
	    Outputs[i + 1] = PredIn[k][i];

	for (j = FirstHidden; j < Nunits; j++)
	{
	    sum = 0.0;
	    for (i = Nconn[j]; i < Nconn[j + 1]; i++)
		sum += Outputs[Conn[i]] * wts[i];
	    if (j < NSunits)
		sum = sigmoid(sum);
	    Outputs[j] = sum;
	    if (j >= FirstOutput)
		PredOut[k][0] = sum;
	}
    }
    for (k = 0; k < NPred; k++)
	*pred++ = PredOut[k][0];
	pred -= NPred;
    free_matrix(PredIn, NPred, Ninputs);
    free_matrix(PredOut, NPred, Noutputs);
}

static void
Build_Net(int ninputs, int nhidden, int noutputs)
{
  Nunits = 1 + ninputs + nhidden + noutputs;
  /*  if (Nunits > N) errmsg("Too many units\n"); */
  Nconn = Calloc(Nunits+1, int);
  Outputs = Calloc(Nunits, double);
  ErrorSums = Calloc(Nunits, double);
  Errors = Calloc(Nunits, double);
  toutputs = Calloc(Nunits, float);
  Ninputs = ninputs;
  FirstHidden = 1 + ninputs;
  FirstOutput = 1 + ninputs + nhidden;
  Noutputs = noutputs;
  Outputs[0] = 1.0;
}



#define EPS 1.0E-80

static double
E(double y, double t)
{
  double  dif, sum = 0;
  if (Entropy) {
    if (t > 0) sum -= t * log((y + EPS) / t);
    if (t < 1) sum -= (1 - t) * log((1 - y + EPS) / (1 - t));
  }
  else {
    dif = y - t;
    sum = dif * dif;
  }
  return (sum);
}
static double 
model5(double y, double t, float stime, int subj)
{
    int i;
    double sum;
    
    sum=-t*y;
    for (i=0;i<Nintervals;i++)
	sum+=stime*exp(Outputs5[subj][FirstOutput][i])/(double)Nintervals;
    return(sum);
}

static void
calcOuts(float *input, int subj, float stime)
{
    int i, k,j;
    double sum;
   for (k=0; k<Nintervals;k++)
    {    
	input[Ninputs-1]= stime*(k+1)/(double)Nintervals;
	
	for (i = 0; i < Ninputs; i++) Outputs5[subj][i + 1][k] = input[i];
	
	for (j = FirstHidden; j < Nunits; j++) {
	    sum = 0.0;
	    for (i = Nconn[j]; i < Nconn[j + 1]; i++)
		sum += Outputs5[subj][Conn[i]][k] * wts[i];
	    if (j < NSunits) sum = sigmoid(sum);
	    Outputs5[subj][j][k] = sum;
	}
    }
    input[Ninputs-1]=stime;
}


static double
modelE(double y, double t, float stime, double alpha)
{
  double sum, tmp;
  if (Model==1)
    sum = exp(y)*stime - y*t;
  else if (Model==2)
    {
      tmp = exp(Alpha)*(y + log(stime));
      if (tmp > 80)
	sum = 999999;
      else
	sum = - t*(y + Alpha + (exp(Alpha)-1)* (log(stime) + y)-
		   log(1 + exp(tmp))) +
	  log(1+ exp(tmp));
    }
  else if (Model==3||Model==6)
    {
  	if (Model==3) alpha=Alpha;
	
    tmp = exp(alpha)*(y+log(stime));
      if(tmp<7.0)
	sum = -t*(alpha - log(stime) -tmp*tmp/2) -
	  (1-t)*log(1-pnorm(tmp,0.0,1.0,1,0));
      else
	sum = 999999;
      /*Rprintf("%f %f %f Model E\n",sum, tmp,Alpha);fflush(stdout); */
    }
  else if (Model==4)
    {
      tmp = exp(Alpha)*(y+log(stime));
      if(tmp>80)
	  sum = 999999;
      else
	  sum = -t*(y+Alpha + (exp(Alpha)-1)*(y + log(stime))) +
	      exp(tmp);
    }
  return (sum);
}

static void
fpass(float *input, float *goal, float wx, float stime,int subj, int all)
{
  int     i, j;
  double  sum, t, thisError;

  for (i = 0; i < Ninputs; i++) Outputs[i + 1] = input[i];

  for (j = FirstHidden; j < Nunits; j++) {
    sum = 0.0;
    for (i = Nconn[j]; i < Nconn[j + 1]; i++)
      sum += Outputs[Conn[i]] * wts[i];
    if (j < NSunits) sum = sigmoid(sum);
    Outputs[j] = sum;
  }
  if (all==0) return;

  if (Softmax) {
    sum = 0.0;
    /* avoid overflows by re-normalizing */
    t = Outputs[FirstOutput];
    for (i = FirstOutput + 1; i < Nunits; i++)
      if (Outputs[i] > t) t = Outputs[i];
    for (i = FirstOutput; i < Nunits; i++) {
      Probs[i] = exp(Outputs[i] - t);
      sum += Probs[i];
    }
    thisError = 0.0;
    for (i = FirstOutput; i < Nunits; i++) {
      Probs[i] = Probs[i] / sum;
      t = goal[i - FirstOutput];
      if(Censored) {
	if(t == 1) thisError += Probs[i];
      } else if (t > 0) {
	if(Probs[i] > 0) TotalError -= wx * t * log(Probs[i]);
	else TotalError += wx * 1000;
      }
    }
    if (Censored) {
      if (thisError > 0) TotalError -= wx * log(thisError);
      else TotalError += wx * 1000;   
    }
  } 
  else 
    if (Model)
	for (i = FirstOutput; i < FirstOutput+1; i++)
	    if (Model==5)
		TotalError += wx * model5(Outputs[i],
					  goal[i - FirstOutput],stime,subj); 
	    else
		TotalError += wx * modelE(Outputs[i],goal[i - FirstOutput],
					  stime,Outputs[Nunits-1]/varWt); 
    else
      for (i = FirstOutput; i < Nunits; i++)
	TotalError += wx * E(Outputs[i], goal[i - FirstOutput]);
}
static void
phfpass(float **input, int trim)
{
    int             i, j, k, l;
    double          sum, maxeta, mineta;

    maxeta = -1000000.0;
    mineta = 1000000.0;
    if (phflag==2)
    {
	for (k = 0; k < NTrain; k++)
	{
	    for (l = 0; l < Nzone; l++)
	    {
		input[k][Ninputs-1] = Uzone[l];
		for (i = 0; i < Ninputs; i++)
		    Outputs[i + 1] = input[k][i];
		for (j = FirstHidden; j < Nunits; j++)
		{
		    sum = 0.0;
		    for (i = Nconn[j]; i < Nconn[j + 1]; i++)
			sum += Outputs[Conn[i]] * wts[i];
		    if (j < NSunits)
			sum = sigmoid(sum);
		    Outputs[j] = sum;
		    if (j < FirstOutput)
			HiddenOut[k][j - FirstHidden][l] = sum;
		    else
		    {
			phTrainOut[k][0][l] = sum;
			if (sum > maxeta)
			    maxeta = sum;
			if (sum < mineta)
			    mineta = sum;
		    }
		}
	    }
	}
	for (k = 0; k < NTrain; k++)
	{
	    for (l = 0; l < Nzone; l++)
	    {
		if (trim) phTrainOut[k][0][l] -= maxeta;
		phTrainOut[k][1][l] = exp(phTrainOut[k][0][l]);
	    }
	    
	}
    }    
    else if (phflag==1)
    {
	for (k = 0; k < NTrain; k++)
	{
	    for (i = 0; i < Ninputs; i++)
		Outputs[i + 1] = input[k][i];
	    for (j = FirstHidden; j < Nunits; j++)
	    {
		sum = 0.0;
		for (i = Nconn[j]; i < Nconn[j + 1]; i++)
		    sum += Outputs[Conn[i]] * wts[i];
		if (j < NSunits)
		    sum = sigmoid(sum);
		Outputs[j] = sum;
		if (j < FirstOutput)
		    HiddenOut[k][j - FirstHidden][0] = sum;
		else
		{
		    phTrainOut[k][0][0] = sum;
		    if (sum > maxeta)
			maxeta = sum;
		    if (sum < mineta)
			mineta = sum;
		}
	    }
	}
	for (k = 0; k < NTrain; k++)
	{
	    if (trim) phTrainOut[k][0][0] -= maxeta;
	    phTrainOut[k][1][0] = exp(phTrainOut[k][0][0]);
	}	
    }
}

static double
sigmoid_prime(double value)
{
  return (value * (1.0 - value));
}

static double
sigmoid_prime_prime(double value)
{
  return (value * (1.0 - value) * (1.0 - 2.0 * value));
}

static void
bpass(float *goal, float wx, float stime,int subj)
{
  int     i, j, cix, k;
  double  sum, denom,tmp,alpha;

  if (Softmax) {
    if (Censored) {
      denom = 0.0;
      for (i = FirstOutput; i < Nunits; i++)
	if (goal[i - FirstOutput] == 1) denom += Probs[i];
      for (i = FirstOutput; i < Nunits; i++) {
	ErrorSums[i] = Probs[i];
	if (goal[i - FirstOutput] == 1) ErrorSums[i] -= Probs[i]/denom;
      }
    } else {
      sum = 0.0;
      for (i = FirstOutput; i < Nunits; i++)
	sum += goal[i - FirstOutput];
      for (i = FirstOutput; i < Nunits; i++)
	ErrorSums[i] = sum * Probs[i] - goal[i - FirstOutput];
    }
  } else if (Entropy)
    for (i = FirstOutput; i < Nunits; i++)
      ErrorSums[i] = Outputs[i] - goal[i - FirstOutput];
  else if (Model==1){ /*exp*/
    /*Rprintf("Model==1");fflush(stdout);*/
    for (i = FirstOutput; i < Nunits; i++){
      ErrorSums[i] = exp(Outputs[i])*stime - goal[i - FirstOutput];
    }}
  else if (Model==2) /* llog */
    {
      for (i = FirstOutput; i < Nunits; i++){
	tmp = exp(Alpha)*(Outputs[i]+ log(stime));
	ErrorSums[i] = exp(Alpha) *( exp(tmp)  - goal[i - FirstOutput])
	  /(1+ exp(tmp));
	dAlpha 
	  -= goal[i-FirstOutput] + 
	  tmp * (goal[i-FirstOutput] - exp(tmp))/(1+ exp(tmp));
   }}
   else if (Model==3||Model==6) /* lnorm */
     {
	 if (Model==3) alpha=Alpha;
	 else
	     alpha=Outputs[Nunits-1]/varWt;
	 i = FirstOutput;
	 sum = exp(alpha) *(Outputs[i] + log(stime));
	   if (sum < 7.0){
	     ErrorSums[i] = goal[i-FirstOutput] * exp(alpha) * sum +
	       (1 - goal[i-FirstOutput]) * exp(alpha) * exp(-0.5 * sum * sum)/
	       sqrt(2 * PI)/(1-pnorm(sum, 0.0, 1.0,1,0));
	     tmp 
	       = -goal[i-FirstOutput]*(1 - sum * sum) +
	       (1-goal[i-FirstOutput]) * sum * exp(-0.5 * sum * sum)/
	       sqrt(2*PI)/(1-pnorm(sum, 0.0, 1.0,1,0));}
	   else
	   {
	       if (Model==3)
	       {
		   ErrorSums[i] = goal[i-FirstOutput]* exp(alpha) * sum +
		       (1 - goal[i-FirstOutput]) * exp(alpha) * sum;
		   tmp 
		       = -goal[i-FirstOutput]*(1 - sum * sum) +
		       (1-goal[i-FirstOutput]) * sum * sum;
	       }	 
	       else
	       {	
		   ErrorSums[i] = 999999.0;
		   tmp=999999.0;
	       }
	 }
	 if (Model==6)
	     ErrorSums[i+1] =tmp/varWt;
	 else
	     dAlpha+=tmp;
	 
     }
  else if (Model==4) /* Weibull */
    {
      for (i = FirstOutput; i < Nunits; i++)
	{
	  ErrorSums[i] = exp(Alpha) * (exp(exp(Alpha)* (Outputs[i] + log(stime)))-
				      goal[i - FirstOutput]);
	  dAlpha 
	    -=goal[i-FirstOutput] + exp(Alpha)* (Outputs[i] + log(stime))*
	    (goal[i-FirstOutput] - exp(exp(Alpha)* (Outputs[i] + log(stime))));
	}
    }
  else if (Model==5) /* free hazard: need to pass bits back*/
  {
        for (i = FirstOutput; i < Nunits; i++)
	  ErrorSums[i] = -goal[i-FirstOutput] ;
/*	Rprintf("%d %f %f bpass \n",FirstOutput,ErrorSums[FirstOutput],goal[0]);
	fflush(stdout);*/
	
  }
  else
    for (i = FirstOutput; i < Nunits; i++) {
      ErrorSums[i] = 2 * (Outputs[i] - goal[i - FirstOutput]);
      if (i < NSunits)
	ErrorSums[i] *= sigmoid_prime(Outputs[i]);
    }
  for (i = FirstHidden; i < FirstOutput; i++) ErrorSums[i] = 0.0;
  
  for (j = Nunits - 1; j >= FirstHidden; j--) {
    Errors[j] = ErrorSums[j];
    if (j < FirstOutput)
      Errors[j] *= sigmoid_prime(Outputs[j]);
    for (i = Nconn[j]; i < Nconn[j + 1]; i++) {
      cix = Conn[i];
      ErrorSums[cix] += Errors[j] * wts[i];
      Slopes[i] += wx * Errors[j] * Outputs[cix];
    }
  }
  /*for (i =0;i<Nunits;i++) Rprintf("%d %f\n",i, Slopes[i]);fflush(stdout);*/
  if (Model==5)
     for (k=0;k<Nintervals;k++)
      {
	  ErrorSums[FirstOutput]=stime*exp(Outputs5[subj][FirstOutput][k])/
(double)Nintervals;
	  for (i = FirstHidden; i < FirstOutput; i++) ErrorSums[i] = 0.0;
	  for (j = Nunits - 1; j >= FirstHidden; j--) {
	      Errors[j] = ErrorSums[j];
	      if (j < FirstOutput)
		  Errors[j] *= sigmoid_prime(Outputs5[subj][j][k]);
	      for (i = Nconn[j]; i < Nconn[j + 1]; i++) {
		  cix = Conn[i];
		  ErrorSums[cix] += Errors[j] * wts[i];
		  Slopes[i] += wx * Errors[j] * Outputs5[subj][cix][k];	  
	      }
	  }
      }    
}

void
survdfunc(double *p, double *df, double *fp)
{
  int     i, j;
  double  sum1;

  if(Model>1&&Model<5) Alpha = p[Nweights];
  for (i = 0; i < Nweights; i++) wts[i] = p[i];
  for (j = 0; j < Nweights; j++) Slopes[j] = 2 * Decay[j] * wts[j];
  TotalError = 0.0;
  dAlpha=0.0;
  for (i = 0; i < NTrain; i++) {
      if (Model==5)  /* set up outputs5[i,1:100] to pick up later*/
	  calcOuts(TrainIn[i], i, TrainStime[i]);
   fpass(TrainIn[i], TrainOut[i], Weights[i], TrainStime[i],i,1);
    bpass(TrainOut[i], Weights[i], TrainStime[i],i);
  }
  sum1 = 0.0;
  for (i = 0; i < Nweights; i++) sum1 += Decay[i] * p[i] * p[i];
  *fp = TotalError + sum1;
  for (j = 0; j < Nweights; j++) df[j] = Slopes[j];
  Epoch++;
  if (Model>1&&Model<5) df[Nweights] = dAlpha;
}

static double
fminfn(double *p)
{
  int     i, pat;
  double  sum1;
  if (phflag==0)
  {
      for (i = 0; i < Nweights; i++) wts[i] = p[i];
      if (Model>1 &&Model<5) Alpha = p[Nweights];
      TotalError = 0.0;
      for (i = 0; i < NTrain; i++)
      {
	  
	  if (Model==5)  /* set up outputs5[i,1:Nintervals] to pick up later*/
	      calcOuts(TrainIn[i], i, TrainStime[i]);
	  fpass(TrainIn[i], TrainOut[i], Weights[i], TrainStime[i],
		i,1);
      }
      sum1 = 0.0;
      for (i = 0; i < Nweights; i++) sum1 += Decay[i] * p[i] * p[i];
      Epoch++;
     /* Rprintf("%f %f here\n",TotalError,wts[0]);fflush(stdout);*/
      return (TotalError + sum1);
  }
  else
  {
      sum1 = 0.0;
      for (i = 0; i < Nweights; i++)
      {
	  wts[i] = p[i];
	  sum1 -= wts[i] * wts[i] * Decay[i];
      }
      
      phfpass(TrainIn, 1);
      if (phflag==2)
      {
	  for (i = 0; i < Nzone; i++)
	      ptexp[i] = 0.0;
	  for (pat = 0; pat < NTrain; pat++)
	  {
	      for (i=0; i <= Zone[pat];i++)
		  ptexp[i] += phTrainOut[pat][1][i];
	      
	      if (Status[pat])	/* dead */
	      {
		  sum1 += phTrainOut[pat][0][Zone[pat]];
		  sum1 -= log(ptexp[Zone[pat]] + 1e-30);
	      }
	  }
      }
      else if(phflag==1)
      {
	  psexp = 0.0;
	  for (pat = 0; pat < NTrain; pat++)
	  {
	      psexp += phTrainOut[pat][1][0];
	      if (Status[pat])	/* dead */
	      {
		  sum1 += phTrainOut[pat][0][0];
		  sum1 -= log(psexp + 1e-30);
	      }
	  }
      }
      return (-sum1);
  }
}

static void
fmingr(double *p, double *df)
{
  int     i, j, pat, skip, k;

  for (i = 0; i < Nweights; i++) wts[i] = p[i];
  if (phflag==0)
  {
      if (Model>1&&Model<5) Alpha = p[Nweights];
      for (j = 0; j < Nweights; j++) Slopes[j] = 2 * Decay[j] * wts[j];
      TotalError = 0.0;
      dAlpha = 0.0;
      for (i = 0; i < NTrain; i++) {
	  if (Model==5)  /* set up Outputs5[i][][1:Nintervals]*/
	      calcOuts(TrainIn[i], i, TrainStime[i]);
	  fpass(TrainIn[i], TrainOut[i], Weights[i], TrainStime[i],
		 i,1);
	  bpass(TrainOut[i], Weights[i], TrainStime[i], i);
      }
      for (j = 0; j < Nweights; j++) df[j] = Slopes[j];
      Epoch++;
      if(Model>1&&Model<5) df[Nweights] = dAlpha;
/*Rprintf("%f %f %f\n",Alpha, dAlpha, TotalError);*/
  }
  else
  {
      phfpass(TrainIn, 1);
      
      if (phflag==2)
      {
	  for (i =0; i < Ninputs; i++)
	  {
	      tmpx[i] = 0.0;
	      for (k = 0; k < Nzone; k++)
		  ptexpx[i][k] = 0.0;
	  }
	  for (j = 0; j < Nhidden; j++)
	  {
	      tmp[j] = 0.0;
	      for (k = 0; k < Nzone; k++)
		  ptexpy[j][k] = 0.0;
	      for (i = 0; i <= Ninputs; i++)
	      {
		  for (k = 0; k < Nzone; k++)
		      ptexpwyx[i][j][k] = 0.0;
		  tmpij[i][j] = 0.0;
	      }
	  }
	  for (k = 0; k < Nzone; k++)
	      ptexp[k] = 0.0;
	  
	  skip=0;
	  if (((Ninputs + 2)*Nhidden+1) < Nweights)
	      skip=1;
	  for (pat = 0; pat < NTrain; pat++)
	  {
	      for (k = 0; k <= Zone[pat];k++)
		  ptexp[k] += phTrainOut[pat][1][k];
	      if (skip)
	      {
		  for (k = 0; k <= Zone[pat]; k++)
		  {
		      TrainIn[pat][Ninputs-1] = Uzone[k];
		      for (i = 0; i < Ninputs; i++)
			  ptexpx[i][k] += phTrainOut[pat][1][k]*TrainIn[pat][i];
		  }
		  TrainIn[pat][Ninputs-1] = Uzone[Zone[pat]];
		  if (Status[pat])
		      for (i = 0; i < Ninputs; i++)
		      { /*dead*/
			  tmpx[i] += TrainIn[pat][i];
			  tmpx[i] -= ptexpx[i][Zone[pat]]/ptexp[Zone[pat]];
		      }
	      }
	      for (j = 0; j < Nhidden; j++)
	      {
		  for (k = 0; k <= Zone[pat]; k++)
		  {		
		      ptexpy[j][k] += phTrainOut[pat][1][k] * HiddenOut[pat][j][k];
		      ptexpwyx[0][j][k] += phTrainOut[pat][1][k] *
			  wts[Nconn[FirstOutput] + j + 1] *
			  sigmoid_prime(HiddenOut[pat][j][k]);
		      TrainIn[pat][Ninputs-1] = Uzone[k];
		      for (i = 0; i < Ninputs; i++)
		      {
			  ptexpwyx[i + 1][j][k] += phTrainOut[pat][1][k] *
			      wts[Nconn[FirstOutput] + j + 1] *
			      sigmoid_prime(HiddenOut[pat][j][k]) *
			      TrainIn[pat][i];
		      }
		  }
		  
		  if (Status[pat])	/* dead */
		  {
		      TrainIn[pat][Ninputs-1] = Uzone[Zone[pat]];
		      tmp[j] += HiddenOut[pat][j][Zone[pat]];
		      tmp[j] -= ptexpy[j][Zone[pat]] / ptexp[Zone[pat]];
		      tmpij[0][j] +=
			  wts[Nconn[FirstOutput] + j + 1] *
			  sigmoid_prime(HiddenOut[pat][j][Zone[pat]]);
		      tmpij[0][j] -= ptexpwyx[0][j][Zone[pat]] / ptexp[Zone[pat]];
		      for (i = 0; i < Ninputs; i++)
		      {
			  tmpij[i + 1][j] +=
			      wts[Nconn[FirstOutput] + j + 1] *
			      sigmoid_prime(HiddenOut[pat][j][Zone[pat]]) *
			      TrainIn[pat][i];
			  tmpij[i + 1][j] -= ptexpwyx[i + 1][j][Zone[pat]] / 
			      ptexp[Zone[pat]];
		      }
		  }
	      }
	  }
      }
      else if (phflag==1)
      {	
	  for (i =0; i < Ninputs; i++)
	  {
	      tmpx[i] = 0.0;
	      psexpx[i] = 0.0;
	  }
	  for (j = 0; j < Nhidden; j++)
	  {
	      tmp[j] = 0.0;
	      psexpy[j] = 0.0;
	      for (i = 0; i <= Ninputs; i++)
	      {
		  psexpwyx[i][j] = 0.0;
		  tmpij[i][j] = 0.0;
	      }
	  }
	  psexp = 0.0;
	  skip=0;
	  if (((Ninputs + 2)*Nhidden+1) < Nweights)
	      skip=1;
	  
	  for (pat = 0; pat < NTrain; pat++)
	  {
	      psexp += phTrainOut[pat][1][0];
	      if (skip)
		  for (i = 0; i < Ninputs; i++)
		  {
		      psexpx[i] += phTrainOut[pat][1][0]*TrainIn[pat][i];
		      if (Status[pat])
		      { /*dead*/
			  tmpx[i] += TrainIn[pat][i];
			  tmpx[i] -= psexpx[i]/psexp;
		      }
		  }
	      for (j = 0; j < Nhidden; j++)
	      {
		  psexpy[j] += phTrainOut[pat][1][0] * HiddenOut[pat][j][0];
		  psexpwyx[0][j] += phTrainOut[pat][1][0] *
		      wts[Nconn[FirstOutput] + j + 1] *
		      sigmoid_prime(HiddenOut[pat][j][0]);
		  for (i = 0; i < Ninputs; i++)
		  {
		      psexpwyx[i + 1][j] += phTrainOut[pat][1][0] *
			  wts[Nconn[FirstOutput] + j + 1] *
			  sigmoid_prime(HiddenOut[pat][j][0]) *
			  TrainIn[pat][i];
		  }
		  if (Status[pat])	/* dead */
		  {
		      tmp[j] += HiddenOut[pat][j][0];
		      tmp[j] -= psexpy[j] / psexp;
		      tmpij[0][j] +=
			  wts[Nconn[FirstOutput] + j + 1] *
			  sigmoid_prime(HiddenOut[pat][j][0]);
		      tmpij[0][j] -= psexpwyx[0][j] / psexp;
		      for (i = 0; i < Ninputs; i++)
		      {
			  tmpij[i + 1][j] +=
			      wts[Nconn[FirstOutput] + j + 1] *
			      sigmoid_prime(HiddenOut[pat][j][0]) *
			      TrainIn[pat][i];
			  tmpij[i + 1][j] -= psexpwyx[i + 1][j] / psexp;
		      }
		  }
	      }
	  }
      }
      if (skip)
	  for (i=0; i<Ninputs;i++)
	      df[Nconn[FirstOutput]+Nhidden+i+1] = -tmpx[i];
      for (j = 0; j < Nhidden; j++)
      {
	  df[Nconn[FirstOutput] + j + 1] = -tmp[j];
	  for (i = 0; i <= Ninputs; i++)
	      df[Nconn[j + FirstHidden] + i] = -tmpij[i][j];
      }
      for (j = 0; j <Nweights;j++)
	  df[j]  += Decay[j] * 2.0 * 
	      wts[j];
  }
}
void phtfunc(double *p, double *df, double *fp)
{    
    *fp = fminfn(p);
    fmingr(p, df);
}

void phtfunc2(double *p, double *Hess)
{
    phHessian(p, Hess);
}



static float *vect(int n)
{
  float  *v;

  v = Calloc(n, float);
  if (!v) errmsg("allocation failure in vect()");
  return v;
}

static void free_vect(float *v)
{
   Free(v);
}
static void free_dvect(double *v)
{
	Free(v);
}

static double *dvect(int n)
{
	double *v;
	v = Calloc(n, double);
	if (!v) errmsg("allocation failure in vect()");
	return v;
}
 
static int *ivect(int n)
{
	int *v;
	v = Calloc(n, int);
	if (!v) errmsg("allocation failure in ivect()");
	return v;
}
 
static void free_ivect(int *v)
{
	Free(v);
}


static float **matrix(int nrh, int nch)
{
  int     i;
  float **m;

  m = Calloc((nrh + 1), float *);
  if (!m) errmsg("allocation failure 1 in matrix()");

  for (i = 0; i <= nrh; i++) {
    m[i] = Calloc((nch + 1), float);
    if (!m[i]) errmsg("allocation failure 2 in matrix()");
  }
  return m;
}

static void free_matrix(float **m, int nrh, int nch)
{
  int     i;

  for (i = nrh; i >= 0; i--) Free(m[i]);
  Free(m);
}
static double **dmatrix(int nr, int nc)
{
	int i;
	double **m;
 
	if( nr < 0 || nc < 0)
	    errmsg("allocation failure 0 in dmatrix()");
	m = Calloc((nr+1), double*);
	if (!m) errmsg("allocation failure 1 in dmatrix()");
 
	for(i = 0;i <= nr;i++) {
		m[i] = Calloc((nc+1), double);
		if (!m[i]) errmsg("allocation failure 2 in dmatrix()");
	}
	return m;
}

static double **Lmatrix(int n)
{
  int     i;
  double **m;

  m = Calloc(n, double *);
  if (!m) errmsg("fail1 in Lmatrix()");

  for (i = 0; i < n; i++) {
    m[i] = Calloc((i + 1), double);
    if (!m[i]) errmsg("fail2 in Lmatrix()");
  }
  return m;
}

static void free_Lmatrix(double **m, int n)
{
  int     i;

  for (i = n - 1; i >= 0; i--) Free(m[i]);
  Free(m);
}
static void free_dmatrix(double **m, int nr, int nc)
{
	int i;
 
	for(i = nr;i >= 0;i--) Free(m[i]);
	Free(m);
}
static double ***d3matrix(int nr, int nc, int nd)
{
    int             i,j;
    double       ***m;

    if( nr < 0 || nc < 0 || nd < 0 )
	errmsg("allocation failure 0 in d3matrix()");
    m = Calloc((nr + 1), double **);
    if (!m)
	errmsg("allocation failure 1 in d3matrix()");

    for (i = 0; i <= nr; i++)
    {
	m[i] = Calloc((nc + 1), double *);
	if (!m[i])
	    errmsg("allocation failure 2 in d3matrix()");
	for (j = 0; j <= nc; j++)
	{
	    m[i][j] = Calloc((nd + 1), double);
	    if (!(m[i][j]))
		errmsg("allocation failure 3 in d3matrix()");
	}
    }
    return m;
}
static void 
free_d3matrix(double ***m, int nr, int nc, int nd)
{
    int             i, j;

    for (i = nr; i >= 0; i--)
    {
	for (j = nc; j >= 0; j--)
	    Free(m[i][j]);
	Free(m[i]);
    }
    Free(m);
}
static double ****
d4matrix(int nr, int nc, int nd, int ne)
{
    int             i, j, k;
    double      ****m;

    if( nr < 0 || nc < 0 || nd < 0 || ne < 0 )
	errmsg("allocation failure 0 in d4matrix()");
    m = Calloc((nr + 1), double ***);
    if (!m)
	errmsg("allocation failure 1 in d4matrix()");

    for (i = 0; i <= nr; i++)
    {
	m[i] = Calloc((nc + 1), double **);
	if (!m[i])
	    errmsg("allocation failure 2 in d4matrix()");
	for (j = 0; j <= nc; j++)
	{
	    m[i][j] = Calloc((nd + 1), double *);
	    if (!(m[i][j]))
		errmsg("allocation failure 3 in d4matrix()");
	    for (k = 0; k <= nd; k++)
	    {
		m[i][j][k] = Calloc((ne + 1), double);
		if (!(m[i][j][k]))
		    errmsg("allocation failure 4 in d4matrix()");
	    }
	}
    }
    return m;
}
static void
free_d4matrix(double ****m, int nr, int nc, int nd, int ne)
{
    int             i, j, k;

    for (i = nr; i >= 0; i--)
    {
	for (j = nc; j >= 0; j--)
	{
	    for (k = nd; k >= 0; k--)
		Free(m[i][j][k]);
	    Free(m[i][j]);
	}
	Free(m[i]);
    }
    Free(m);
}
static double *****
d5matrix(int nr, int nc, int nd, int ne, int nf)
{
    int             i, j, k, l;
    double      *****m;

    if( nr < 0 || nc < 0 || nd < 0 || ne < 0 || nf < 0)
	errmsg("allocation failure 0 in d5matrix()");
    m = Calloc((nr + 1), double ****);
    if (!m)
	errmsg("allocation failure 1 in d5matrix()");
    
    for (i = 0; i <= nr; i++)
    {
	m[i] = Calloc((nc + 1), double ***);
	if (!m[i])
	    errmsg("allocation failure 2 in d5matrix()");
	for (j = 0; j <= nc; j++)
	{
	    m[i][j] = Calloc((nd + 1), double **);
	    if (!(m[i][j]))
		errmsg("allocation failure 3 in d5matrix()");
	    for (k = 0; k <= nd; k++)
	    {
		m[i][j][k] = Calloc((ne + 1), double *);
		if (!(m[i][j][k]))
		    errmsg("allocation failure 4 in d5matrix()");
		for (l = 0; l <= ne; l++)
		{
		    m[i][j][k][l] = Calloc((nf + 1), double );
		    if (!(m[i][j][k][l]))
			errmsg("allocation failure 5 in d5matrix()");
		}
	    }
	}
    }
    return m;
}
static void
free_d5matrix(double *****m, int nr, int nc, int nd, int ne, int nf)
{
    int             i, j, k, l;

    for (i = nr; i >= 0; i--)
    {
	for (j = nc; j >= 0; j--)
	{
	    for (k = nd; k >= 0; k--)
	    {
		for (l = ne; l >= 0; l--)
		    Free(m[i][j][k][l]);
		Free(m[i][j][k]);
	    }
	    Free(m[i][j]);
	}
	Free(m[i]);
    }
    Free(m);
}


void
survdovm(int *Nw, double *wts, double *Fmin,
	int *maxit, int *trace, int *mask)
{
  float abstol;
  
  if (Model >0) abstol = -1.0e20;
  else abstol=1.0e-4;
  
  vmmin((int) *Nw, wts, Fmin, (int) *maxit, (int) *trace, mask, abstol);
}
static void dophvm(int *Nw, double *wts, double *Fmin, int *maxit, int *trace, int *mask)
{
    float abstol;

    abstol=1.0e-4;
    vmmin((int)*Nw, wts, Fmin, (int)*maxit, (int)*trace, mask, abstol);
}

static void
phHessian(double *inwts, double *Hess)
{
    int             i, j, k, l, pat, from[2], to[2], hid1, hid0, z;
    double 	   **H, x0, x1, a, b, c;
    for (i = 0; i < Nweights; i++)
	wts[i] = inwts[i];
    phfpass(TrainIn, 1);
    for (i = 0; i < Nunits; i++)
	for (j = Nconn[i]; j < Nconn[i + 1]; j++)
	    Dest[j] = i;
    H = dmatrix(Nweights, Nweights);
    for (i = 0; i < Nweights; i++)
    {
	for (j = 0; j < Nweights; j++)
	    H[i][j] = 0.0;
    }
    if (phflag==2)
    {
	ptexpyy = d3matrix(Nhidden,Nhidden,Nzone);
	ptexpxy = d3matrix(Ninputs,Nhidden,Nzone);
	ptexpxwyx = d4matrix(Ninputs,Ninputs+1,Nhidden,Nzone);
	ptexpxx = d3matrix(Ninputs,Ninputs,Nzone);
	ptexpyx = d3matrix(Ninputs+1, Nhidden,Nzone);
	ptexpwyyx = d4matrix(Ninputs+1, Nhidden, Nhidden,Nzone);
	ptexpwYxx = d4matrix(Ninputs+1, Ninputs+1, Nhidden,Nzone);
	ptexpwyxwyx = d5matrix(Ninputs+1, Ninputs+1, Nhidden, Nhidden,Nzone);
	for (z = 0; z < Nzone; z++)
	{
	    for (j = 0; j < Ninputs; j++)
	    {
		ptexpx[j][z] = 0.0;
		for (i = 0; i < Ninputs; i++)
		    ptexpxx[j][i][z] = 0.0;
		for (i = 0; i < Nhidden; i++)
		    ptexpxy[j][i][z] = 0.0;
	    }
	    for (j = 0; j < Nhidden; j++)
	    {
		ptexpy[j][z] = 0.0;
		for (i = 0; i <= Ninputs; i++)
		{
		    ptexpwyx[i][j][z] = 0.0;
		    ptexpyx[i][j][z] = 0.0;
		    for (k=0;k<Nhidden;k++)
			ptexpwyyx[i][k][j][z] = 0.0;
		    for (k=0;k<=Ninputs;k++)
		    {
			ptexpwYxx[i][k][j][z] = 0.0;
			for (l=0; l < Nhidden; l++)
			    ptexpwyxwyx[i][k][l][j][z] =0.0;
		    }
		    for (k=0;k<Ninputs;k++)
			ptexpxwyx[k][i][j][z] = 0.0;
		}
		for (i = 0; i < Nhidden; i++)
		    ptexpyy[i][j][z] = 0.0;
	    }
	    ptexp[z] = 0.0;
	}
	for (pat = 0; pat < NTrain; pat++)
	{
	    for (z = 0; z <= Zone[pat]; z++)
	    {
		ptexp[z] += phTrainOut[pat][1][z];
		TrainIn[pat][Ninputs-1] = Uzone[z];   
		for (j = 0; j < Ninputs; j++)
		{
		    ptexpx[j][z] += phTrainOut[pat][1][z]*TrainIn[pat][j];
		    for (i = 0; i < Ninputs; i++)
			ptexpxx[j][i][z] += phTrainOut[pat][1][z]*TrainIn[pat][j]*
			    TrainIn[pat][i];
		    for (i = 0; i < Nhidden; i++)
			ptexpxy[j][i][z] += phTrainOut[pat][1][z]*TrainIn[pat][j]*
			    HiddenOut[pat][i][z];
		}
		for (j = 0; j < Nhidden; j++)
		{
		    ptexpy[j][z] += phTrainOut[pat][1][z] * HiddenOut[pat][j][z];
		    for (i = 0; i < Nhidden; i++)
		    {
			ptexpyy[i][j][z] += phTrainOut[pat][1][z] *
			    HiddenOut[pat][i][z] * HiddenOut[pat][j][z];
			ptexpwyxwyx[0][0][j][i][z] +=
			    phTrainOut[pat][1][z] *
			    wts[Nconn[FirstOutput] + j + 1] *
			    wts[Nconn[FirstOutput] + i + 1] *
			    sigmoid_prime(HiddenOut[pat][j][z]) *
			    sigmoid_prime(HiddenOut[pat][i][z]);
			ptexpwyyx[0][j][i][z] += phTrainOut[pat][1][z] *
			    wts[Nconn[FirstOutput] + j + 1] *
			    sigmoid_prime(HiddenOut[pat][j][z]) *
			    HiddenOut[pat][i][z];
			for (k = 0; k < Ninputs; k++)
			{
			    ptexpwyyx[k + 1][j][i][z] += phTrainOut[pat][1][z] *
				wts[Nconn[FirstOutput] + j + 1] *
				sigmoid_prime(HiddenOut[pat][j][z]) *
				HiddenOut[pat][i][z] * TrainIn[pat][k];
			    ptexpwyxwyx[0][k + 1][j][i][z] +=
				phTrainOut[pat][1][z] *
				wts[Nconn[FirstOutput] + j + 1] *
				wts[Nconn[FirstOutput] + i + 1] *
				sigmoid_prime(HiddenOut[pat][j][z]) *
				sigmoid_prime(HiddenOut[pat][i][z]) * 
				TrainIn[pat][k];
			    ptexpwyxwyx[k + 1][0][j][i][z] +=
				phTrainOut[pat][1][z] *
				wts[Nconn[FirstOutput] + j + 1] *
				wts[Nconn[FirstOutput] + i + 1] *
				sigmoid_prime(HiddenOut[pat][j][z]) *
				sigmoid_prime(HiddenOut[pat][i][z]) * 
				TrainIn[pat][k];
			}
		    }
		    ptexpwyx[0][j][z] += phTrainOut[pat][1][z] *
			wts[Nconn[FirstOutput] + j + 1] *
			sigmoid_prime(HiddenOut[pat][j][z]);
		    ptexpyx[0][j][z] += phTrainOut[pat][1][z] *
			sigmoid_prime(HiddenOut[pat][j][z]);
		    ptexpwYxx[0][0][j][z] += phTrainOut[pat][1][z] *
			wts[Nconn[FirstOutput] + j + 1] *
			sigmoid_prime_prime(HiddenOut[pat][j][z]);
		    for (i = 0; i < Ninputs; i++)
		    {
			ptexpwyx[i + 1][j][z] += phTrainOut[pat][1][z] *
			    wts[Nconn[FirstOutput] + j + 1] *
			    sigmoid_prime(HiddenOut[pat][j][z]) *
			    TrainIn[pat][i];
			ptexpyx[i + 1][j][z] += phTrainOut[pat][1][z] *
			    sigmoid_prime(HiddenOut[pat][j][z]) *
			    TrainIn[pat][i];
			ptexpwYxx[0][i + 1][j][z] += phTrainOut[pat][1][z] *
			    wts[Nconn[FirstOutput] + j + 1] *
			    sigmoid_prime_prime(HiddenOut[pat][j][z]) *
			    TrainIn[pat][i];
			ptexpwYxx[i + 1][0][j][z] += phTrainOut[pat][1][z] *
			    wts[Nconn[FirstOutput] + j + 1] *
			    sigmoid_prime_prime(HiddenOut[pat][j][z]) *
			    TrainIn[pat][i];
			ptexpxwyx[i][0][j][z] += phTrainOut[pat][1][z]*
			    TrainIn[pat][i] *
			    wts[Nconn[FirstOutput] + j + 1] *
			    sigmoid_prime(HiddenOut[pat][j][z]);
			for (k = 0; k < Ninputs; k++)
			{
			    ptexpxwyx[i][k+1][j][z] += phTrainOut[pat][1][z]*
				TrainIn[pat][i] *
				wts[Nconn[FirstOutput] + j + 1] *
				sigmoid_prime(HiddenOut[pat][j][z])*
				TrainIn[pat][k];
			    ptexpwYxx[k + 1][i + 1][j][z] += phTrainOut[pat][1][z] *
				wts[Nconn[FirstOutput] + j + 1] *
				sigmoid_prime_prime(HiddenOut[pat][j][z]) *
				TrainIn[pat][i] * TrainIn[pat][k];
			    for (l = 0; l < Nhidden; l++)
				ptexpwyxwyx[k + 1][i + 1][j][l][z] +=
				    phTrainOut[pat][1][z] *
				    wts[Nconn[FirstOutput] + j + 1] *
				    wts[Nconn[FirstOutput] + l + 1] *
				    sigmoid_prime(HiddenOut[pat][j][z]) *
				    sigmoid_prime(HiddenOut[pat][l][z]) *
				    TrainIn[pat][i] * TrainIn[pat][k];
			}
		    }
		}
	    }
	    TrainIn[pat][Ninputs-1] = Uzone[Zone[pat]];
	    if (Status[pat])	/* dead */
	    {
		z = Zone[pat];
		for (i = 0; i < Nweights; i++)
		{
		    from[0] = Conn[i];
		    to[0] = Dest[i];
		    for (j = i; j < Nweights; j++)
		    {
			from[1] = Conn[j];
			to[1] = Dest[j];
			if ((to[0] < FirstOutput) & (to[1] < FirstOutput))
			{		/* to both  hidden units, from both
					   input or bias */
			    hid0 = to[0] - FirstHidden;
			    hid1 = to[1] - FirstHidden;
			    H[i][j] += (ptexpwyx[from[0]][hid0][z] *
					ptexpwyx[from[1]][hid1][z] / ptexp[z] / ptexp[z]) -
				(ptexpwyxwyx[from[0]][from[1]][hid0][hid1][z]
				 / ptexp[z]);
			    if (to[0] == to[1])
			    {
				if (from[0] == 0)
				    x0 = 1.0;
				else
				    x0 = TrainIn[pat][from[0] - 1];
				if (from[1] == 0)
				    x1 = 1.0;
				else
				    x1 = TrainIn[pat][from[1] - 1];
				H[i][j] += wts[Nconn[FirstOutput] + hid0 + 1] *
				    sigmoid_prime_prime(HiddenOut[pat][hid0][z])
				    * x1 * x0
				    - (ptexpwYxx[from[0]][from[1]][hid0][z]
				       / ptexp[z]);
			    }
			}
			else if ((to[0]< FirstOutput) & (to[1] == FirstOutput))
			{
			    /* to[0] hidden, to[1] output,
			       from[1] hidden or bias or skip and bias
			       can be ignored*/
			    if (from[0] == 0)
				x0 = 1.0;
			    else
				x0 = TrainIn[pat][from[0] - 1];
			    hid0 = to[0] - FirstHidden;
			    if (from[1] > 0)
			    {
				if (from[1] <= Ninputs)
				{
				    a = ptexpx[from[1]-1][z];
				    b = ptexpxwyx[from[1]-1][from[0]][hid0][z];
				}
				else
				{
				    a = ptexpy[from[1] - FirstHidden][z];
				    b = ptexpwyyx[from[0]][hid0]
					[from[1]-FirstHidden][z];
				}
				H[i][j] += a *  ptexpwyx[from[0]][hid0][z]
				    / (ptexp[z] * ptexp[z]) - b / ptexp[z];
				if (to[0] == from[1])
				    H[i][j] += sigmoid_prime(HiddenOut[pat][hid0][z])
					* x0 -
					(ptexpyx[from[0]][hid0][z] / ptexp[z]);
			    }
			}
			else if ((to[0] == FirstOutput) &
				 (to[1] == FirstOutput))
			{/* from both hidden or bias or skip and bias 
			    can be ignored*/
			    if ((from[0] > 0) & (from[1] > 0))
			    {
				if (from[0] <= Ninputs)
				{ 
				    a = ptexpx[from[0]-1][z];
				    if (from[1] <= Ninputs)
					c = ptexpxx[from[0]-1][from[1]-1][z];
				    else
					c = ptexpxy[from[0]-1][from[1]-FirstHidden][z];
				}
				else
				{
				    a = ptexpy[from[0]-FirstHidden][z];
				    if (from[1] <= Ninputs) 
					c = ptexpxy[from[1]-1][from[0]-FirstHidden][z];
				    else
					c = ptexpyy[from[0]-FirstHidden]
					    [from[1]-FirstHidden][z];
				}
				if (from[1] <= Ninputs)
				    b = ptexpx[from[1]-1][z];
				else
				    b = ptexpy[from[1]-FirstHidden][z];
				H[i][j] += a * b / ptexp[z]/ptexp[z] - c/ptexp[z];
			    }
			}
		    }
		}
	    }

	}
	for (i=0;i<Nweights;i++)
	    for (j=0;j<Nweights;j++)
	    {
		if (i==j) H[i][j] -= 2*Decay[i]; 
		*Hess++ = -H[i][j];
	    }
	fflush(stdout);
	free_dmatrix(H,Nweights,Nweights);
	free_d3matrix(ptexpyy,Nhidden,Nhidden, Nzone);
	free_d3matrix(ptexpyx,Ninputs+1, Nhidden,Nzone);
	free_d3matrix(ptexpxy,Ninputs, Nhidden, Nzone);
	free_d4matrix(ptexpxwyx,Ninputs, Ninputs+1, Nhidden, Nzone);
	free_d3matrix(ptexpxx,Ninputs, Ninputs, Nzone);
	free_d4matrix(ptexpwyyx,Ninputs+1, Nhidden, Nhidden, Nzone);
	free_d4matrix(ptexpwYxx,Ninputs+1, Ninputs+1, Nhidden, Nzone);
	free_d5matrix(ptexpwyxwyx,Ninputs+1,Ninputs+1,Nhidden,Nhidden,Nzone); 
    }
    else if (phflag==1)
    {
	psexpyy = dmatrix(Nhidden,Nhidden);
	psexpxy = dmatrix(Ninputs,Nhidden);
	psexpxwyx = d3matrix(Ninputs,Ninputs+1,Nhidden);
	psexpxx = dmatrix(Ninputs,Ninputs);
	psexpyx = dmatrix(Ninputs+1, Nhidden);
	psexpwyyx = d3matrix(Ninputs+1, Nhidden, Nhidden);
	psexpwYxx = d3matrix(Ninputs+1, Ninputs+1, Nhidden);
	psexpwyxwyx = d4matrix(Ninputs+1, Ninputs+1, Nhidden, Nhidden);
	for (j = 0; j < Ninputs; j++)
	{
	    psexpx[j] = 0.0;
	    for (i = 0; i < Ninputs; i++)
		psexpxx[j][i] = 0.0;
	    for (i = 0; i < Nhidden; i++)
		psexpxy[j][i] = 0.0;
	}
	for (j = 0; j < Nhidden; j++)
	{
	    psexpy[j] = 0.0;
	    for (i = 0; i <= Ninputs; i++)
	    {
		psexpwyx[i][j] = 0.0;
		psexpyx[i][j] = 0.0;
		for (k=0;k<Nhidden;k++)
		    psexpwyyx[i][k][j] = 0.0;
		for (k=0;k<=Ninputs;k++)
		{
		    psexpwYxx[i][k][j] = 0.0;
		    for (l=0; l < Nhidden; l++)
			psexpwyxwyx[i][k][l][j] =0.0;
		}
		for (k=0;k<Ninputs;k++)
		    psexpxwyx[k][i][j] = 0.0;
	    }
	    for (i = 0; i < Nhidden; i++)
		psexpyy[i][j] = 0.0;
	}
	psexp = 0.0;
	for (pat = 0; pat < NTrain; pat++)
	{
	    psexp += phTrainOut[pat][1][0];
	    for (j = 0; j < Ninputs; j++)
	    {
		psexpx[j] += phTrainOut[pat][1][0]*TrainIn[pat][j];
		for (i = 0; i < Ninputs; i++)
		    psexpxx[j][i] += phTrainOut[pat][1][0]*TrainIn[pat][j]*TrainIn[pat][i];
		for (i = 0; i < Nhidden; i++)
		    psexpxy[j][i] += phTrainOut[pat][1][0]*TrainIn[pat][j]*
			HiddenOut[pat][i][0];
	    }
	    for (j = 0; j < Nhidden; j++)
	    {
		psexpy[j] += phTrainOut[pat][1][0] * HiddenOut[pat][j][0];
		for (i = 0; i < Nhidden; i++)
		{
		    psexpyy[i][j] += phTrainOut[pat][1][0] *
			HiddenOut[pat][i][0] * HiddenOut[pat][j][0];
		    psexpwyxwyx[0][0][j][i] +=
			phTrainOut[pat][1][0] *
			wts[Nconn[FirstOutput] + j + 1] *
			wts[Nconn[FirstOutput] + i + 1] *
			sigmoid_prime(HiddenOut[pat][j][0]) *
			sigmoid_prime(HiddenOut[pat][i][0]);
		    psexpwyyx[0][j][i] += phTrainOut[pat][1][0] *
			wts[Nconn[FirstOutput] + j + 1] *
			sigmoid_prime(HiddenOut[pat][j][0]) *
			HiddenOut[pat][i][0];
		    for (k = 0; k < Ninputs; k++)
		    {
			psexpwyyx[k + 1][j][i] += phTrainOut[pat][1][0] *
			    wts[Nconn[FirstOutput] + j + 1] *
			    sigmoid_prime(HiddenOut[pat][j][0]) *
			    HiddenOut[pat][i][0] * TrainIn[pat][k];
			psexpwyxwyx[0][k + 1][j][i] +=
			    phTrainOut[pat][1][0] *
			    wts[Nconn[FirstOutput] + j + 1] *
			    wts[Nconn[FirstOutput] + i + 1] *
			    sigmoid_prime(HiddenOut[pat][j][0]) *
			    sigmoid_prime(HiddenOut[pat][i][0]) * TrainIn[pat][k];
			psexpwyxwyx[k + 1][0][j][i] +=
			    phTrainOut[pat][1] [0]*
			    wts[Nconn[FirstOutput] + j + 1] *
			    wts[Nconn[FirstOutput] + i + 1] *
			    sigmoid_prime(HiddenOut[pat][j][0]) *
			    sigmoid_prime(HiddenOut[pat][i][0]) * TrainIn[pat][k];
		    }
		}
		psexpwyx[0][j] += phTrainOut[pat][1][0] *
		    wts[Nconn[FirstOutput] + j + 1] *
		    sigmoid_prime(HiddenOut[pat][j][0]);
		psexpyx[0][j] += phTrainOut[pat][1][0] *
		    sigmoid_prime(HiddenOut[pat][j][0]);
		psexpwYxx[0][0][j] += phTrainOut[pat][1][0] *
		    wts[Nconn[FirstOutput] + j + 1] *
		    sigmoid_prime_prime(HiddenOut[pat][j][0]);
		for (i = 0; i < Ninputs; i++)
		{
		    psexpwyx[i + 1][j] += phTrainOut[pat][1][0] *
			wts[Nconn[FirstOutput] + j + 1] *
			sigmoid_prime(HiddenOut[pat][j][0]) *
			TrainIn[pat][i];
		    psexpyx[i + 1][j] += phTrainOut[pat][1][0] *
			sigmoid_prime(HiddenOut[pat][j][0]) *
			TrainIn[pat][i];
		    psexpwYxx[0][i + 1][j] += phTrainOut[pat][1][0] *
			wts[Nconn[FirstOutput] + j + 1] *
			sigmoid_prime_prime(HiddenOut[pat][j][0]) *
			TrainIn[pat][i];
		    psexpwYxx[i + 1][0][j] += phTrainOut[pat][1][0] *
			wts[Nconn[FirstOutput] + j + 1] *
			sigmoid_prime_prime(HiddenOut[pat][j][0]) *
			TrainIn[pat][i];
		    psexpxwyx[i][0][j] += phTrainOut[pat][1][0]*TrainIn[pat][i] *
			wts[Nconn[FirstOutput] + j + 1] *
			sigmoid_prime(HiddenOut[pat][j][0]);
		    for (k = 0; k < Ninputs; k++)
		    {
			psexpxwyx[i][k+1][j] += phTrainOut[pat][1][0]*TrainIn[pat][i] *
			    wts[Nconn[FirstOutput] + j + 1] *
			    sigmoid_prime(HiddenOut[pat][j][0])*
			    TrainIn[pat][k];
			psexpwYxx[k + 1][i + 1][j] += phTrainOut[pat][1][0] *
			    wts[Nconn[FirstOutput] + j + 1] *
			    sigmoid_prime_prime(HiddenOut[pat][j][0]) *
			    TrainIn[pat][i] * TrainIn[pat][k];
			for (l = 0; l < Nhidden; l++)
			    psexpwyxwyx[k + 1][i + 1][j][l] +=
				phTrainOut[pat][1][0] *
				wts[Nconn[FirstOutput] + j + 1] *
				wts[Nconn[FirstOutput] + l + 1] *
				sigmoid_prime(HiddenOut[pat][j][0]) *
				sigmoid_prime(HiddenOut[pat][l][0]) *
				TrainIn[pat][i] * TrainIn[pat][k];
		    }
		}
	    }
	    if (Status[pat])	/* dead */
	    {
		for (i = 0; i < Nweights; i++)
		{
		    from[0] = Conn[i];
		    to[0] = Dest[i];
		    for (j = i; j < Nweights; j++)
		    {
			from[1] = Conn[j];
			to[1] = Dest[j];
			if ((to[0] < FirstOutput) & (to[1] < FirstOutput))
			{		/* to both  hidden units, from both
					   input or bias */
			    hid0 = to[0] - FirstHidden;
			    hid1 = to[1] - FirstHidden;
			    H[i][j] += (psexpwyx[from[0]][hid0] *
					psexpwyx[from[1]][hid1] / psexp / psexp) -
				(psexpwyxwyx[from[0]][from[1]][hid0][hid1]
				 / psexp);
			    if (to[0] == to[1])
			    {
				if (from[0] == 0)
				    x0 = 1.0;
				else
				    x0 = TrainIn[pat][from[0] - 1];
				if (from[1] == 0)
				    x1 = 1.0;
				else
				    x1 = TrainIn[pat][from[1] - 1];
				H[i][j] += wts[Nconn[FirstOutput] + hid0 + 1] *
				    sigmoid_prime_prime(HiddenOut[pat][hid0][0])
				    * x1 * x0
				    - (psexpwYxx[from[0]][from[1]][hid0]
				       / psexp);
			    }
			}
			else if ((to[0] < FirstOutput) & (to[1] == FirstOutput))
			{		/* to[0] hidden, to[1] output,
					   from[1] hidden or bias or skip and bias can be ignored*/
			    if (from[0] == 0)
				x0 = 1.0;
			    else
				x0 = TrainIn[pat][from[0] - 1];
			    hid0 = to[0] - FirstHidden;
			    if (from[1] > 0)
			    {
				if (from[1] <= Ninputs)
				{
				    a = psexpx[from[1]-1];
				    b = psexpxwyx[from[1]-1][from[0]][hid0];
				}
				else
				{
				    a = psexpy[from[1] - FirstHidden];
				    b = psexpwyyx[from[0]][hid0][from[1]-FirstHidden];
				}
				H[i][j] += a *  psexpwyx[from[0]][hid0]
				    / (psexp * psexp) - b / psexp;
				if (to[0] == from[1])
				    H[i][j] += sigmoid_prime(HiddenOut[pat][hid0][0])
					* x0 -
					(psexpyx[from[0]][hid0] / psexp);
			    }
			}
			else if ((to[0] == FirstOutput) &
				 (to[1] == FirstOutput))
			{		/* from both hidden or bias or skip and bias can be ignored*/
			    if ((from[0] > 0) & (from[1] > 0))
			    {
				if (from[0] <= Ninputs)
				{ 
				    a = psexpx[from[0]-1];
				    if (from[1] <= Ninputs)
					c = psexpxx[from[0]-1][from[1]-1];
				    else
					c = psexpxy[from[0]-1][from[1]-FirstHidden];
				}
				else
				{
				    a = psexpy[from[0]-FirstHidden];
				    if (from[1] <= Ninputs) 
					c = psexpxy[from[1]-1][from[0]-FirstHidden];
				    else
					c = psexpyy[from[0]-FirstHidden][from[1]-FirstHidden];
				}
				if (from[1] <= Ninputs)
				    b = psexpx[from[1]-1];
				else
				    b = psexpy[from[1]-FirstHidden];
				H[i][j] += a * b / psexp/psexp - c/psexp;
			    }
			}
		    }
		}
	    }
	    
	}
	for (i=0;i<Nweights;i++)
	    for (j=0;j<Nweights;j++)
	    {
		if (i==j) H[i][j] -= 2*Decay[i]; 
		*Hess++ = -H[i][j];
	    }
	fflush(stdout);
	free_dmatrix(H,Nweights,Nweights);
	free_dmatrix(psexpyy,Nhidden,Nhidden);
	free_dmatrix(psexpyx,Ninputs+1, Nhidden);
	free_dmatrix(psexpxx,Ninputs, Ninputs);
	free_dmatrix(psexpxy,Ninputs, Nhidden);
	free_d3matrix(psexpxwyx,Ninputs, Ninputs+1,Nhidden);
	free_d3matrix(psexpwyyx,Ninputs+1, Nhidden, Nhidden);
	free_d3matrix(psexpwYxx,Ninputs+1, Ninputs+1, Nhidden);
	free_d4matrix(psexpwyxwyx,Ninputs+1, Ninputs+1, Nhidden, Nhidden);
    }
}

void 
phdoit(int *ntr, float *train, int *status, int *Nw, double *wts, double *Fmin, int *maxit, int *trace, int *mask, double *haz, double *Hess,
        int *dohaz, int *dohess)
{
    int             pat, i;
    double          ps;
    
    set_phtrain(ntr, train, status);

    dophvm(Nw, wts, Fmin, maxit, trace, mask);
   if (*dohaz)
    {
    	phfpass(TrainIn, 0);
	for (pat = 0; pat < NTrain; pat++)
	    haz[pat] = 0.0;
	psexp = 0.0;
	for (pat = 0; pat < NTrain; pat++)
	{
	    psexp += phTrainOut[pat][1][0];
	    if (Status[pat])
	    {
		ps = 1 / psexp;
		for (i = 0; i <= pat; i++)
		    haz[i] += ps;
	    }
	}
    }
    if (*dohess)
	phHessian(wts, Hess);
}


void 
phtdoit(int *ntr, float *train, int *status, int *Nw, double *wts, double *Fmin, int *maxit, int *trace, int *mask, double *haz, double *Hess, int *dohaz, int *dohess, int *nzone, int *zone, float *uzone)
{
    int             pat, i;
    double          ps;    
    set_phttrain(ntr, train, status, nzone, zone, uzone);
    dophvm(Nw, wts, Fmin, maxit, trace, mask);
    
    if (*dohaz)
    {
    	phfpass(TrainIn, 0);
	
	for (pat = 0; pat < NTrain; pat++)
	    haz[pat] = 0.0;
	for (i = 0; i < Nzone; i++)
	    ptexp[i] = 0.0;
	for (pat = 0; pat < NTrain; pat++)
	{
	    for (i = 0; i <= Zone[pat]; i++)
		ptexp[i] += phTrainOut[pat][1][i];
	    if (Status[pat])
	    {
		ps = 1 / ptexp[Zone[pat]];
		for (i = 0; i <= pat; i++)
		    haz[i] += ps;
	    }
	}
    }
    if (*dohess)
	phHessian(wts, Hess);
    
}

typedef unsigned char Boolean;

#define false 0

#define stepredn	0.2
#define acctol		0.0001
#define reltest		10.0
/*#define abstol 		1.0e-4*/
#define reltol 		1.0e-8
#define REPORT		10


/*  BFGS variable-metric method, based on Pascal code
in J.C. Nash, `Compact Numerical Methods for Computers', 2nd edition,
converted by p2c then re-crafted by B.D. Ripley */

static void
vmmin(int n0, double *b, double *Fmin, int maxit, int trace, int *mask,
float abstol)
{
  Boolean accpoint, enough;
  double  *g, *t, *X, *c, **B;
  int     count, funcount, gradcount;
  double  f, gradproj;
  int     i, j, ilast, iter = 0;
  double  s, steplength;
  double  D1, D2;
  int     n, *l;

  l = Calloc(n0, int);
  if (!l) errmsg("allocation failure in dovm()");
  n = 0;
  for (i = 0; i < n0; i++)
    if (mask[i]) l[n++] = i;

  g = dvect(n0);
  t = dvect(n);
  X = dvect(n);
  c = dvect(n);
  B = Lmatrix(n);
  f = fminfn(b);
  if (trace) {
    Rprintf("initial  value %f \n", f);
    fflush(stdout);
  }
  {
    *Fmin = f;
    funcount = gradcount = 1;
    fmingr(b, g);
    iter++;
    ilast = gradcount;

    do {
      if (ilast == gradcount) {
	for (i = 0; i < n; i++) {
	  for (j = 0; j < i; j++)
	    B[i][j] = 0.0;
	  B[i][i] = 1.0;
	}
      }

      for (i = 0; i < n; i++) {
	X[i] = b[l[i]];
	c[i] = g[l[i]];
      }
      gradproj = 0.0;
      for (i = 0; i < n; i++) {
	s = 0.0;
	for (j = 0; j <= i; j++) s -= B[i][j] * g[l[j]];
	for (j = i + 1; j < n; j++) s -= B[j][i] * g[l[j]];
	t[i] = s;
	gradproj += s * g[l[i]];
      }

      if (gradproj < 0.0) {	/* search direction is downhill */
	steplength = 1.0;
	accpoint = false;
	do {
	  count = 0;
	  for (i = 0; i < n; i++) {
	    b[l[i]] = X[i] + steplength * t[i];
	    if (reltest + X[i] == reltest + b[l[i]])	/* no change */
	      count++;
	  }
	  if (count < n) {
	    f = fminfn(b);
	    funcount++;
	    accpoint = (f <= *Fmin + gradproj * steplength * acctol);

	    if (!accpoint) {
	      steplength *= stepredn;
	    }
	  }
	} while (!(count == n || accpoint));
	enough = (f > abstol) && (f < (1.0 - reltol) * (*Fmin));
	/* stop if value if small or if relative change is low */
	if (!enough) count = n;
	if (count < n) {	/* making progress */
	  *Fmin = f;
	  fmingr(b, g);
	  gradcount++;
	  iter++;
	  D1 = 0.0;
	  for (i = 0; i < n; i++) {
	    t[i] = steplength * t[i];
	    c[i] = g[l[i]] - c[i];
	    D1 += t[i] * c[i];
	  }
	  if (D1 > 0) {
	    D2 = 0.0;
	    for (i = 0; i < n; i++) {
	      s = 0.0;
	      for (j = 0; j <= i; j++) s += B[i][j] * c[j];
	      for (j = i + 1; j < n; j++) s += B[j][i] * c[j];
	      X[i] = s;
	      D2 += s * c[i];
	    }
	    D2 = 1.0 + D2 / D1;
	    for (i = 0; i < n; i++) {
	      for (j = 0; j <= i; j++)
		B[i][j] += (D2 * t[i] * t[j] - X[i] * t[j] - t[i] * X[j]) / D1;
	    }
	  } else {		/* D1 < 0 */
	    ilast = gradcount;
	  }
	} else {		/* no progress */
	  if (ilast < gradcount) {
	    count = 0;
	    ilast = gradcount;
	  }
	}
      } else {			/* uphill search */
	count = 0;
	if (ilast == gradcount) count = n;
	else ilast = gradcount;
	/* Resets unless has just been reset */
      }
      if (iter % REPORT == 0 && trace) {
	Rprintf("iter%4d value %f\n", iter, f);
	fflush(stdout);
      }
      if (iter >= maxit) break;
      if (gradcount - ilast > 2 * n)
	ilast = gradcount;	/* periodic restart */
    } while (count != n || ilast != gradcount);
  }
  if (trace) {
    Rprintf("final  value %f \n", *Fmin);
    if (iter < maxit) Rprintf("converged\n");
    else Rprintf("stopped after %i iterations\n", iter);
  }

  free_dvect(g);
  free_dvect(t);
  free_dvect(X);
  free_dvect(c);
  free_Lmatrix(B, n);
  Free(l);
}


static double **H, *h, *h1, **w;

static void
pHessian(float *input, float *goal, float wx, float stime, int subj)
{
  int     i, to1, to2, from1, from2, j, j1, j2, first1, first2;
  double  out, s, sum1, sum2, t, tmp, tot, P;

  fpass(input, goal, 1.0, stime,subj,0);
  if (Model==5)
      calcOuts(input,subj,stime);

  bpass(goal, 1.0, stime, subj);

  /* Formulae from Ripley (1996, p.152) */
  if (Softmax) {
    for (i = 0; i < Nunits; i++) {
      sum1 = 0.0; sum2 = 0.0; tot = 0.0; P = 0.0;
      for (j = FirstOutput; j < Nunits; j++) {
	sum1 += w[i][j] * Probs[j];
	t = goal[j - FirstOutput];
	P += t * Probs[j];
	sum2 += w[i][j] * Probs[j] * t;
	tot += t;
      }
      h[i] = sum1; h1[i] = sum2/P;
    }
    if(Censored) tot = 1;
    for (to1 = 0; to1 < Nunits; to1++)
      for (j1 = Nconn[to1]; j1 < Nconn[to1 + 1]; j1++) {
	from1 = Conn[j1];
	first1 = (to1 < FirstOutput);
	for (to2 = 0; to2 < Nunits; to2++)
	  for (j2 = Nconn[to2]; j2 < Nconn[to2 + 1]; j2++)
	    if (j2 <= j1) {
	      from2 = Conn[j2];
	      first2 = (to2 < FirstOutput);
	      if ((!first1) && (!first2)) {  /* both -> output */
		if(Censored) {
		  tmp = -Probs[to1] * Probs[to2] * 
		    (1 - goal[to1 - FirstOutput] * goal[to2 - FirstOutput]/P/P);
		  if (to1 == to2) tmp += Probs[to1] * (1 - goal[to1 - FirstOutput]/P);
		  H[j1][j2] += wx * (tmp * Outputs[from1] * Outputs[from2]);	  
		} else {
		    tmp = -Probs[to1] * Probs[to2];
		    if (to1 == to2) tmp += Probs[to1];
		    H[j1][j2] += wx * tot * (tmp * Outputs[from1] * Outputs[from2]);
		}
	      } else if (first1 && first2) {  /* both -> hidden */
		sum1 = sum2 = 0.0;
		for (i = FirstOutput; i < Nunits; i++) {
		  sum1 += Errors[i] * w[to1][i];
		  tmp = w[to1][i] * w[to2][i] * Probs[i];
		  if(Censored) tmp *= (1 - goal[i - FirstOutput]/P);
		  sum2 += tmp;
		}
		if(Censored) {
		  sum2 += - h[to1] * h[to2] + h1[to1] * h1[to2];
		  s = sigmoid_prime(Outputs[to1]) * sigmoid_prime(Outputs[to2])
		    * sum2;
		} else  {
		  sum2 -= h[to1] * h[to2];
		  s = sigmoid_prime(Outputs[to1]) * sigmoid_prime(Outputs[to2])
		    * tot * sum2;
		}
		if (to1 == to2)
		  s += sigmoid_prime_prime(Outputs[to1]) * sum1;
		H[j1][j2] += wx * (s * Outputs[from1] * Outputs[from2]);
	      } else {  /* one -> hidden, one -> output */
		if (to1 < to2) {
		  tmp = w[to1][to2] - h[to1];
		  if(Censored) tmp += goal[to2 - FirstOutput]/P *
				 (h1[to1] - w[to1][to2]);
		  H[j1][j2] += wx * (Outputs[from1] * sigmoid_prime(Outputs[to1])
				     * (Outputs[from2] * Probs[to2] * tmp * tot
					+ ((to1 == from2) ? Errors[to2] : 0)));
		} else {
		  tmp = w[to2][to1] - h[to2];
		  if(Censored) tmp += goal[to1 - FirstOutput]/P *
				 (h1[to2] - w[to2][to1]);
		  H[j1][j2] += wx * (Outputs[from2] * sigmoid_prime(Outputs[to2])
				     * (Outputs[from1] * Probs[to1] * tmp * tot
					+ ((to2 == from1) ? Errors[to1] : 0)));
		}
	      }
	    }
      }
  } else if (Model) ModelHess(input,goal,wx, stime, subj);
  else
  {  /* Not softmax */
    for (i = FirstOutput; i < Nunits; i++) {
      out = Outputs[i];
      s = sigmoid_prime(out);
      t = goal[i - FirstOutput];
      if (Linout) h[i] = 2;
      else if (Entropy) h[i] = out * (1 - out);
      else h[i] = sigmoid_prime_prime(out) * 2 * (out - t) + 2 * s * s;
    }
    for (to1 = 0; to1 < Nunits; to1++)
      for (j1 = Nconn[to1]; j1 < Nconn[to1 + 1]; j1++) {
	from1 = Conn[j1];
	first1 = (to1 < FirstOutput);
	for (to2 = 0; to2 < Nunits; to2++)
	  for (j2 = Nconn[to2]; j2 < Nconn[to2 + 1]; j2++)
	    if (j2 <= j1) {
	      from2 = Conn[j2];
	      first2 = (to2 < FirstOutput);
	      if ((!first1) && (!first2)) {  /* both -> output */
		if (to1 == to2)
		  H[j1][j2] += wx * (h[to1] * Outputs[from1] * Outputs[from2]);
	      }
	      else if (first1 && first2) {  /* both -> hidden */
		sum1 = sum2 = 0.0;
		for (i = FirstOutput; i < Nunits; i++) {
		  sum1 += Errors[i] * w[to1][i];
		  sum2 += w[to1][i] * w[to2][i] * h[i];
		}
		s = sigmoid_prime(Outputs[to1]) * sigmoid_prime(Outputs[to2])
		  * sum2;
		if (to1 == to2)
		  s += sigmoid_prime_prime(Outputs[to1]) * sum1;
		H[j1][j2] += wx * (s * Outputs[from1] * Outputs[from2]);
	      }
	      else {  /* one -> hidden, one -> output */
		if (to1 < to2) {
		  H[j1][j2] += wx * (Outputs[from1] * sigmoid_prime(Outputs[to1])
				     * (Outputs[from2] * w[to1][to2] * h[to2]
					+ ((to1 == from2) ? Errors[to2] : 0)));
		}
		else {
		  H[j1][j2] += wx * (Outputs[from2] * sigmoid_prime(Outputs[to2])
				     * (Outputs[from1] * w[to2][to1] * h[to1]
					+ ((to2 == from1) ? Errors[to1] : 0)));
		}
	      }
	    }
      }
  }
}

static void ModelHess(float *input, float *goal, float wx, float stime, int subj)
{  
    int i, to1, to2, from1, from2, j1, j2, first1, first2, j, myout;
    double sum, out,s, sum1, sum2, t, alpha, t1, t3, halphay, s1, s2;
    i = FirstOutput;
    out = Outputs[i];
    t = goal[i - FirstOutput];
    if (Model==1) /*exp */
	h[i] = exp(out)*stime ;
    else if (Model==2) /* llog */
	h[i] = (exp(2*Alpha) * 
		safepow(exp(out) * stime, exp(Alpha)) * (1+t)) /
	    pow((1 + safepow(exp(out) *stime,exp(Alpha))),2.0);
    else  if (Model==3|| Model==6) /* lnorm*/
    {
	if (Model==3)
	    alpha=Alpha;
	else
	    alpha=Outputs[i+1]/varWt;
	sum = exp(alpha) *(out + log(stime));
	h[i] = t* exp(2*alpha) +
	    (1-t)*exp(2*alpha)*dnorm(sum,0.0,1.0,0)/
	    (1-pnorm(sum,0.0,1.0,1,0))*(dnorm(sum,0.0,1.0,0)/(1-pnorm(sum,0.0,1.0,1,0))-sum);
	if (Model==6)
	{
	    t3 = out +log(stime);
	    t1 = exp(alpha)*t3;
	    h[i+1]=(2 * t * t1*t1 + 
		    (1-t)*t1*dnorm(t1,0.0,1.0,0)/(1-pnorm(t1,0.0,1.0,1,0)) *
		    (1-t1*t1 + t1 * dnorm(t1,0.0,1.0,0)/(1-pnorm(t1,0.0,1.0,1,0))))/
		varWt/varWt;
	    halphay=(t * 2 *t1 *exp(alpha) + 
		     (1-t) * exp(alpha) * dnorm(t1,0.0,1.0,0)/
		     (1-pnorm(t1,0.0,1.0,1,0)) *
		     (1 - t1*t1 + t1*dnorm(t1,0.0,1.0,0)/
		      (1-pnorm(t1,0.0,1.0,1,0))))/varWt;
	}
    }
    else if (Model==4) /* Weibull */
	h[i] =  exp(2*Alpha) * exp(exp(Alpha) * out) * 
	    safepow(stime,exp(Alpha));
    else if (Model==5) /* hazard */
    {
	for (j = 0; j < Nintervals; j++)
	{
	  Errors[i]=stime*exp(Outputs5[subj][i][j])/(double)Nintervals;
	    h[i] = stime*exp(Outputs5[subj][i][j])/(double)Nintervals;
	    for (to1 = 0; to1 < Nunits; to1++)
		for (j1 = Nconn[to1]; j1 < Nconn[to1 + 1]; j1++) 
		{
		    from1 = Conn[j1];
		    first1 = (to1 < FirstOutput);
		    for (to2 = 0; to2 < Nunits; to2++)
			for (j2 = Nconn[to2]; j2 < Nconn[to2 + 1]; j2++)
			    if (j2 <= j1) 
			    {
				from2 = Conn[j2];
				first2 = (to2 < FirstOutput);
				if ((!first1) && (!first2))
				{  /* both -> output */
				    if (to1 == to2)
					H[j1][j2] += wx * (h[to1] *
						Outputs5[subj][from1][j] *
				   Outputs5[subj][from2][j]);
				}
				else if (first1 && first2) 
				{  /* both -> hidden */
				    sum1 = Errors[i] * w[to1][i];
				    sum2 = w[to1][i] * w[to2][i] * h[i];
				    s1=sigmoid_prime(Outputs5[subj][to1][j]) *
					sigmoid_prime(Outputs5[subj][to2][j]) *
					sum2;
   if (to1 == to2)
		s2 = sigmoid_prime_prime(Outputs5[subj][to1][j]) * sum1;
   s=s1;
   
   if (to1==to2) s +=s2;
   
			    H[j1][j2] += wx * (s * Outputs5[subj][from1][j] *
					       Outputs5[subj][from2][j]);
				}
				else 
				{  /* one -> hidden, one -> output */
				    if (to1 < to2) 
				    {
				H[j1][j2] += wx * (Outputs5[subj][from1][j] *
				  sigmoid_prime(Outputs5[subj][to1][j])
					   * (Outputs5[subj][from2][j] *
					      w[to1][to2] * h[to2]
					      + ((to1 == from2) ? 
							 Errors[to2] : 0)));
				    }
				    else 
				    {
				H[j1][j2] += wx * (Outputs5[subj][from2][j] *
				   sigmoid_prime(Outputs5[subj][to2][j])
					   * (Outputs5[subj][from1][j] *
						      w[to2][to1] * h[to1]
						      + ((to2 == from1) ? 
							 Errors[to1] : 0)));
				    }
				}
			    }
		    
		}
	    
	    
	}
    
	h[i]=0.0;
	Errors[i]=-t;
	
    }

    for (to1 = 0; to1 < Nunits; to1++)
	for (j1 = Nconn[to1]; j1 < Nconn[to1 + 1]; j1++) 
	{
	    from1 = Conn[j1];
	    first1 = (to1 < FirstOutput);
	    for (to2 = 0; to2 < Nunits; to2++)
		for (j2 = Nconn[to2]; j2 < Nconn[to2 + 1]; j2++)
		    if (j2 <= j1) 
		    {
			from2 = Conn[j2];
			first2 = (to2 < FirstOutput);
			if ((!first1) && (!first2))
			{  /* both -> output */
			    if (to1 == to2)
				H[j1][j2] += wx * (h[to1] * Outputs[from1] *
						   Outputs[from2]);
			    else if (Model==6)
				H[j1][j2] += wx * (halphay *Outputs[from1]*
						   Outputs[from2]);
			}
			else if (first1 && first2) 
			{  /* both -> hidden */
			    sum1 = sum2 = 0.0;
			    for (i = FirstOutput; i < Nunits; i++) {
				sum1 += Errors[i] * w[to1][i];
				sum2 += w[to1][i] * w[to2][i] * h[i];
			    }
			    if (Model==6)
			    {
				i=FirstOutput;
				sum2 += w[to1][i] * w[to2][i+1] * halphay;
				sum2 += w[to1][i+1] * w[to2][i] * halphay;
			    }
			    s = sigmoid_prime(Outputs[to1]) * 
				sigmoid_prime(Outputs[to2])
				* sum2;
			    if (to1 == to2)
				s += sigmoid_prime_prime(Outputs[to1]) * sum1;
			    H[j1][j2] += wx * (s * Outputs[from1] *
					       Outputs[from2]);
			}
			else 
			{  /* one -> hidden, one -> output */
			    if (to1 < to2) 
			    {
				H[j1][j2] += wx * (Outputs[from1] *
						   sigmoid_prime(Outputs[to1])
						   * (Outputs[from2] *
						      w[to1][to2] * h[to2]
						      + ((to1 == from2) ? 
							 Errors[to2] : 0)));
				if (Model==6)
				{
				    if (to2==FirstOutput) myout=to2+1;
				    else myout=to2-1; 
				    H[j1][j2]+= wx*Outputs[from1] *
					sigmoid_prime(Outputs[to1])
					* Outputs[from2] *
					   w[to1][myout] * halphay;
				}		    
			    }
			    else 
			    {
				H[j1][j2] += wx * (Outputs[from2] *
						   sigmoid_prime(Outputs[to2])
						   * (Outputs[from1] *
						      w[to2][to1] * h[to1]
						      + ((to2 == from1) ? 
							 Errors[to1] : 0)));
				if (Model==6)
				{
				    if (to1==FirstOutput) myout=to1+1;
				    else myout=to1-1; 
				    H[j1][j2]+= wx*Outputs[from1] *
					sigmoid_prime(Outputs[to2])
					* Outputs[from2] *
					   w[to2][myout] * halphay;
				}		    
			    }
			}
		    }
}
/* fix up rest of the Hessian for this patient */

}


#define max9(a,b) a>b?a:b
#define min9(a,b) a<b?a:b

void
survnnHessian(double *inwts, double *Hess)
{
  int     i, j, sizeH;

  for (i = 0; i < Nweights; i++) wts[i] = inwts[i];
  if (Model>1&&Model<5)
      sizeH = Nweights+1;
  else
      sizeH = Nweights;
  H = Lmatrix(sizeH);
  h = dvect(Nunits); h1 = dvect(Nunits);
  w = dmatrix(Nunits, Nunits);
  for (i = 0; i < Nweights; i++)
    for (j = 0; j <= i; j++) H[i][j] = 0.0;
  for (j = FirstOutput; j < Nunits; j++)
    for (i = FirstHidden; i < FirstOutput; i++) w[i][j] = 0.0;
  for (j = FirstOutput; j < Nunits; j++)
    for (i = Nconn[j]; i < Nconn[j + 1]; i++) w[Conn[i]][j] = wts[i];
   for (i = 0; i < NTrain; i++) {
    pHessian(TrainIn[i], TrainOut[i], Weights[i], TrainStime[i], i);
  }
  for (i = 0; i < Nweights; i++) H[i][i] += 2 * Decay[i];

  /* set up row and column for alpha if required */
   if (Model > 1&&Model<5)
  {
      dAlpha=0.0;
      for (i = 0; i < Nweights; i++) Slopes[i] = 0.0;
      for (i = 0; i < NTrain; i++)
      {
	  fpass(TrainIn[i], TrainOut[i], 1.0, TrainStime[i],1,0);
	  bpass2(TrainOut[i], 1.0, TrainStime[i]);
      }

      for (i = 0; i < Nweights; i++)
	  H[Nweights][i] = Slopes[i];
      H[Nweights][Nweights] = dAlpha;
  }

  for (i = 0; i < sizeH; i++)
    for (j = 0; j < sizeH; j++) *Hess++ = H[max9(i, j)][min9(i, j)];
  free_Lmatrix(H, sizeH);
  free_dvect(h);free_dvect(h1);
  free_dmatrix(w, Nunits, Nunits);
}
static void
bpass2(float *goal, float wx, float stime)
{
    int     i, j, cix;
    double  t, fv, t1, t2, t3;
    
    t = goal[0];
    fv = Outputs[FirstOutput];
    
    if (Model==2) /* llog */
    {
	t1 = safepow(exp(fv)* stime,exp(Alpha));
	t2 = safepow(exp(fv)* stime, 2*exp(Alpha));
	t3 = fv +log(stime);
	ErrorSums[FirstOutput] = - (exp(Alpha) * (t + t1 * (t-1)  - t2  - t3 *
						  exp(Alpha) * t1*(1+t)))/  
	    (1+ t1)/ (1+ t1);
	dAlpha -= wx * exp(Alpha) /(1+t1) * (t3 * (t-t1) - exp(Alpha)* t3*t3*t1
	     - (exp(Alpha) * t3 * (t-t1) *t3 * t1)/(1+t1));
    }
    else if (Model==3) /* lnorm */
    {
	t3 = fv +log(stime);
	t1 = exp(Alpha)*t3;
	ErrorSums[FirstOutput] = t * 2 *t1 *exp(Alpha) + 
	    (1-t) * exp(Alpha) * dnorm(t1,0.0,1.0,0)/(1-pnorm(t1,0.0,1.0,1,0)) *
	    (1 - t1*t1 + t1*dnorm(t1,0.0,1.0,0)/(1-pnorm(t1,0.0,1.0,1,0)));
	dAlpha		  += 2 * t * t1*t1 + 
	    (1-t)*t1*dnorm(t1,0.0,1.0,0)/(1-pnorm(t1,0.0,1.0,1,0)) *
	    (1-t1*t1 + t1 * dnorm(t1,0.0,1.0,0)/(1-pnorm(t1,0.0,1.0,1,0)));
    }
    else if (Model==4) /* Weibull */
    {
	t3 = fv +log(stime);
	t1 = safepow(exp(fv)* stime,exp(Alpha));
	ErrorSums[FirstOutput] = -t * exp(Alpha) + exp(Alpha)*t1 + 
	    exp(2*Alpha) * t3 * t1;
	dAlpha -= wx * (t*exp(Alpha) * t3  - exp(Alpha) * t3 * t1 -
			exp(2*Alpha)*t3 * t3 * t1);
    }
    for (i = FirstHidden; i < FirstOutput; i++) ErrorSums[i] = 0.0;
    
    for (j = Nunits - 1; j >= FirstHidden; j--) {
	Errors[j] = ErrorSums[j];
	if (j < FirstOutput)
	    Errors[j] *= sigmoid_prime(Outputs[j]);
	for (i = Nconn[j]; i < Nconn[j + 1]; i++) {
	    cix = Conn[i];
	    ErrorSums[cix] += Errors[j] * wts[i];
	    Slopes[i] += wx * Errors[j] * Outputs[cix];
	}
    }
    /*for (i =0;i<Nunits;i++) Rprintf("%d %f\n",i, Slopes[i]);fflush(stdout);*/
}

static double safepow(double x, double y)
{
  double z;
  if (y < 1e-20) return(1.0);
   if(x < 1e-200) return(0.0);
  z = y*log(x);
   if (z < -200) return(0.0);
  if (z > 200) return(1e200);
  return(exp(z));
}

