################################################################################
#                                                                              #
#                            multiplex.R                                       #
#                                                                              #
# R script for illustrating the analysis of multiplex data in RSiena           #
# using Vanina Torlo's data set                                                #
#                                                                              #
# Written by Christian Steglich and Tom Snijders                               #
# Version March 18, 2016                                                       #
#                                                                              #
################################################################################

# Attach RSiena commands:
library(RSiena)

# Set working directory to where the data are:
setwd("C:\\Users\\tom.snijders\\Documents\\Siena\\website\\lab-multiplex") # or wherever
list.files()

# Read the data sets for the assignment:
friendship.net1 <- as.matrix(read.table("MBA-Friendship1.txt"))
friendship.net2 <- as.matrix(read.table("MBA-Friendship2.txt"))
advice.net1 <- as.matrix(read.table("MBA-Advice1.txt"))
advice.net2 <- as.matrix(read.table("MBA-Advice2.txt"))

# Identify dependent variables for the analysis;
advice <- sienaNet(array(c(advice.net1,advice.net2), dim=c(70,70,2)))
friendship <- sienaNet(array(c(friendship.net1, friendship.net2), dim=c(70,70,2)))

# Combine data for the analysis:
ExampleData <- sienaDataCreate(friendship,advice)

# Get effects table for model specification:
ExampleEffects <- getEffects(ExampleData)

# Generate initial descriptive outputfile:
print01Report(ExampleData, modelname="Multiplex-illustration")

# Take a look at the generated output file "Multiplex-illustration.out"
# to see how RSiena interpreted the directives provided.
#
# The specifically "multiplex" effects as given in the manual can be found
# by inspecting:
effectsDocumentation(ExampleEffects)
#
# From here on proceed "as usual", e.g., as follows:

# transitivity, transitivity*reciprocity, and 3-cycle effects
# for the evolution of advice seeking, and that of friendship:
ExampleEffects <- includeEffects(ExampleEffects,transTrip,transRecTrip,cycle3,name="advice")
ExampleEffects <- includeEffects(ExampleEffects,transTrip,transRecTrip,cycle3,name="friendship")

# First we analyse only tie-level dependencies.
# Include main effects of the dependent networks on each other's evolution:
ExampleEffects <- includeEffects(ExampleEffects,crprod,name="advice",interaction1="friendship")
ExampleEffects <- includeEffects(ExampleEffects,crprod,name="friendship",interaction1="advice")
# Inspect effects object:
ExampleEffects

# Create model object:
ExampleAlgo <- sienaAlgorithmCreate(projname="Mplex-illustr")

# Estimate model:
ExampleResults <- siena07(ExampleAlgo,data=ExampleData,effects=ExampleEffects)
ExampleResults
# If the previous run has not led to a converged estimation, then carry on:
ExampleResults <- siena07(ExampleAlgo,data=ExampleData,effects=ExampleEffects,
							prevAns=ExampleResults)
ExampleResults

# Results indicate strong effect of advice on friendship
# and a weaker, but still strong, effect of friendship on advice.
# What does it mean?
# Evaluation function effects model both maintenance of old ties and creation of new ones.
# Let us differentiate between maintenance and creation.
#
# Estimate the corresponding endowment and creation effects instead to test whether the
# found effect is one of maintaining exisiting friendship ties (which would would be
# stabilised by advice seeking) or an effect of creating new ties based on advice seeking.

ExampleEffects <- includeEffects(ExampleEffects,crprod,
   name="friendship",interaction1="advice",type="endow")
ExampleEffects <- includeEffects(ExampleEffects,crprod,
   name="friendship",interaction1="advice",type="creation")
ExampleEffects <- includeEffects(ExampleEffects,crprod,
   name="friendship",interaction1="advice",type="eval",include=FALSE)

# For symmetry reasoning, do the same for the 'opposite effect' of friendship on
# advice seeking:

ExampleEffects <- includeEffects(ExampleEffects,crprod,
   name="advice",interaction1="friendship",type="endow")
ExampleEffects <- includeEffects(ExampleEffects,crprod,
   name="advice",interaction1="friendship",type="creation")
ExampleEffects <- includeEffects(ExampleEffects,crprod,
   name="advice",interaction1="friendship",type="eval",include=FALSE)
ExampleEffects

# Now estimate the model:
ExampleResults <- siena07(ExampleAlgo,data=ExampleData,effects=ExampleEffects)
ExampleResults
# If the previous run has not led to a converged estimation, then carry on:
ExampleResults <- siena07(ExampleAlgo,data=ExampleData,effects=ExampleEffects,
							prevAns=ExampleResults)
ExampleResults

save.image("multiplex-results.RData")

# What can be seen is that there is a significant creation (and no endowment) effect
# of friendship on advice seeking, and a significant endowment (but no creation) effect
# of advice seeking on friendship.
#
# Bottomline: Friendship leads to new advice seeking;
# advice seeking stabilises existing friendship.
# Note the increased standard errors.

# Now also include actor-level effects.
# We specify indegree-indegree and outdegree-outdegree effects,
# with parameter 1 (no square root versions).
ExampleEffects <- setEffect(ExampleEffects,inPopIntn,name="advice",
						interaction1="friendship",parameter=1)
ExampleEffects <- setEffect(ExampleEffects,inPopIntn,name="friendship",
						interaction1="advice",parameter=1)
ExampleEffects <- setEffect(ExampleEffects,outActIntn,name="advice",
						interaction1="friendship",parameter=1)
ExampleEffects <- setEffect(ExampleEffects,outActIntn,name="friendship",
						interaction1="advice",parameter=1)
ExampleEffects
# Estimate while using the previous estimates
# (except of course for the newly included effects) as starting values:
ExampleResults <- siena07(ExampleAlgo,data=ExampleData,effects=ExampleEffects,
							prevAns=ExampleResults)
ExampleResults

# This nicely shows the multilevel aspect:
# For the actor level, the degrees show a negative association.
# For the tie level, there is a positive association,
# which appears stronger when the actor level is controlled for.

# Let us test whether the difference between the creation and endowment effects
# for the dyadic dependence are significant. This is done with function Wald.RSiena,
# see Section 8.1 in the manual, and
?Wald.RSiena

# First test the creation-maintenance difference
# in the dyadic effect of advice on friendship:
# We first construct a vector fulfilling the role of A in the help page:
tested <- rep(0,20)
tested[7] <- 1
tested[8] <- -1
tested
Wald.RSiena(tested, ExampleResults)
# The difference is not significant.

# Next test the creation-maintenance difference
# in the dyadic effect of friendship on advice:
tested <- rep(0,20)
tested[17] <- 1
tested[18] <- -1
tested
Wald.RSiena(tested, ExampleResults)
# Here the difference is significant.
# For reliably testing these hypotheses, we would need
# n3=3000 in sienaAlgorithmCreate.

# Further analyses can study triadic effects.

# See Tom A.B. Snijders, Alessandro Lomi, and Vanina Jasmine Torlo (2013).
# A model for the multiplex dynamics of two-mode and one-mode networks,
# with an application to employment preference, friendship, and advice.
# Social Networks, 35, 265-276.