############################################################
# R script for illustrating the analysis of bipartite data #
############################################################

# load RSiena commands:
library(RSiena)

# set working directory to where the data are:
setwd("<put your directory name here>")
list.files()

# read the friendship data:
friendship1 <- as.matrix(read.table("Glasgow-1-net.dat"))
friendship2 <- as.matrix(read.table("Glasgow-2-net.dat"))

# now demographic info:
demographics <- as.matrix(read.table("Glasgow-sex-birthyear.dat"))

# now the bipartite data:
leisure1 <- as.matrix(read.table("GL-1-lsr.dat"))
leisure2 <- as.matrix(read.table("GL-2-lsr.dat"))
music1 <- as.matrix(read.table("GL-1-mus.dat"))
music2 <- as.matrix(read.table("GL-2-mus.dat"))
drugs1 <- as.matrix(read.table("GL-1-drg.dat"))
drugs2 <- as.matrix(read.table("GL-2-drg.dat"))

# recode into meaningful affiliation info:
leisure1[leisure1 %in% c(2,3,4)] <- 0
# leisure1[leisure1 == 2] <- 1
leisure2[leisure2 %in% c(2,3,4)] <- 0
# leisure2[leisure2 == 2] <- 1
drugs1[drugs1 %in% c(1)] <- 0
drugs1[drugs1 %in% c(2,3,4)] <- 1
drugs2[drugs2 %in% c(1)] <- 0
drugs2[drugs2 %in% c(2,3,4)] <- 1

# also recode valued friendship data & identify missings: 
friendship1[friendship1 == 2] <- 1
friendship1[friendship1 == 9] <- NA
friendship2[friendship2 == 2] <- 1
friendship2[friendship2 == 9] <- NA

# Note that demographic info contains just one missing birth year.
# Because we don't use this variable (but sex only), no recoding here.

# find out number of nodes in nodesets:
nrpupils <- dim(leisure1)[1]
nrleisureItems <- dim(leisure1)[2]
nrmusicItems <- dim(music1)[2]
nrdrugsItems <- dim(drugs1)[2]

# define different node sets:
pupils <- sienaNodeSet(nrpupils, nodeSetName="pupils")
leisureItems <- sienaNodeSet(nrleisureItems, nodeSetName="leisureItems")
musicItems <- sienaNodeSet(nrmusicItems, nodeSetName="musicItems")
drugsItems <- sienaNodeSet(nrdrugsItems, nodeSetName="drugsItems")

# identify dependent variables for the analysis;
# start with bipartite network:
leisure <- sienaNet(array(c(leisure1,leisure2),dim=c(nrpupils,nrleisureItems,2)),
  "bipartite",nodeSet=c("pupils","leisureItems"))
music <- sienaNet(array(c(music1,music2),dim=c(nrpupils,nrmusicItems,2)),
  "bipartite",nodeSet=c("pupils","musicItems"))
drugs <- sienaNet(array(c(drugs1,drugs2),dim=c(nrpupils,nrdrugsItems,2)),
  "bipartite",nodeSet=c("pupils","drugsItems"))
# now add the normal network:
friendship <- sienaNet(array(c(friendship1, friendship2),dim=c(nrpupils,nrpupils,2)),
  nodeSet="pupils")

# include exogenous predictor variables:
sex.F <- coCovar(demographics[,1],nodeSet="pupils")  
  
# combine data for the analysis:
bipData <- sienaDataCreate(friendship,leisure,music,drugs,sex.F,
  nodeSets=list(pupils,leisureItems,musicItems,drugsItems))

# get effects table for model specification:
bipEffects <- getEffects(bipData)

# generate initial descriptive outputfile:
print01Report(bipData, modelname="Bipartite-illustration")

# Take a look at the generated output file "Bipartite-illustration.out"
# to see how RSiena interpreted the directives provided.

# Specify the model:
bipEffects <- includeEffects(bipEffects,transTrip,name="friendship")
bipEffects <- includeEffects(bipEffects,from,name="friendship",interaction1="leisure")
bipEffects <- includeEffects(bipEffects,from,name="friendship",interaction1="music")
bipEffects <- includeEffects(bipEffects,from,name="friendship",interaction1="drugs")
bipEffects <- includeEffects(bipEffects,cycle4,name="leisure")
bipEffects <- includeEffects(bipEffects,cycle4,name="music")
bipEffects <- includeEffects(bipEffects,cycle4,name="drugs")
bipEffects <- includeEffects(bipEffects,to,name="leisure",interaction1="friendship")
bipEffects <- includeEffects(bipEffects,to,name="music",interaction1="friendship")
bipEffects <- includeEffects(bipEffects,to,name="drugs",interaction1="friendship")

# Bypass bug related to initial values for bipartite network rates:
bipEffects <- setEffect(bipEffects,Rate,name="leisure",initialValue=1.0,period=1,type="rate")
bipEffects <- setEffect(bipEffects,Rate,name="music",initialValue=1.0,period=1,type="rate")
bipEffects <- setEffect(bipEffects,Rate,name="drugs",initialValue=1.0,period=1,type="rate")

# create model object:
bipModel <- sienaModelCreate(useStdInits=FALSE,projname='bipartite-Glasgow-results')

# estimate model:
bipResults <- siena07(bipModel,data=bipData,effects=bipEffects,batch=FALSE,verbose=FALSE)

# store results for possible later use:
save.image("bipResults.RData")