######################################################
#              Lab-CoEvolution.R                     #
#                                                    #
# R script for performing the analyses reported in   #
# Steglich, Snijders & West (2006; SSW)              #
# plus a robustness check of the results             #
# upon addition of the 'quad' effect.                # 
# Written by Christian Steglich,                     #
# with some additions by Tom Snijders.               #
# Version Sept 24, 2014                              #
#                                                    #
######################################################

# The results obtained by this script deliver the same qualitative conclusions
# as reported in SSW, but quantitative details differ a bit. 
# This is due to several changes that were applied to both software and data:
#  - Original analyses were performed in February 2005 with the SIENA software 
#    version 2.0u; a documentation of changes to the software since then 
#    can be found in the manuals for Siena 3.2 and RSiena 
#    (both available at http://www.stats.ox.ac.uk/~snijders/siena/).
# -  The quadratic shape effect was not used in SSW
#    (because at that time we were not aware of its importance).
#  - Original analyses were performed on a preliminary version of the 
#    Teenage Health and Lifestyle Study data, which does not fully coincide 
#    with the public release. More documentation on the data set can, again, 
#    be found at the Siena website.

# load RSiena commands:
library(RSiena)

# set working directory to where the data are:
setwd("C:\\Users\\tom.snijders\\Documents\\Siena\\website\\lab-slidesCoEvolution")
list.files()

# Read data sets:
load("Glasgow-friendship.RData")  # friendship networks
load("Glasgow-demographic.RData") # for gender data
load("Glasgow-substances.RData")  # for alcoholo data
load("Glasgow-lifestyle.RData")   # for music listening data
load("Glasgow-selections.RData")  # for 129 respondents used in SSW
# See what we have now:
ls()

# Define music scales as in SSW:
rockscale <- matrix(nrow=length(age),ncol=3)
rownames(rockscale) <- rownames(music1)
rockscale[,1] <- rowSums(music1[,colnames(music1) %in% c("rock","indie","heavy","grunge")])
rockscale[,2] <- rowSums(music2[,colnames(music2) %in% c("rock","indie","heavy","grunge")])
rockscale[,3] <- rowSums(music3[,colnames(music3) %in% c("rock","indie","heavy","grunge")])
elitescale <- matrix(nrow=length(age),ncol=3)
rownames(elitescale) <- rownames(music1)
elitescale[,1] <- rowSums(music1[,colnames(music1) %in% c("folk","jazz","classical")])
elitescale[,2] <- rowSums(music2[,colnames(music2) %in% c("folk","jazz","classical")])
elitescale[,3] <- rowSums(music3[,colnames(music3) %in% c("folk","jazz","classical")])
chartscale <- matrix(nrow=length(age),ncol=3)
rownames(chartscale) <- rownames(music1)
chartscale[,1] <- rowSums(music1[,colnames(music1) %in% c("techno","chart","dance","rave")])
chartscale[,2] <- rowSums(music2[,colnames(music2) %in% c("techno","chart","dance","rave")])
chartscale[,3] <- rowSums(music3[,colnames(music3) %in% c("techno","chart","dance","rave")])

# Recode valued friendship to binary friendship:
friendship.1[friendship.1==2] <- 1
friendship.2[friendship.2==2] <- 1
friendship.3[friendship.3==2] <- 1

# Identify dependent network variable:
friendship <- sienaNet(array(c(friendship.1[selection129,selection129],
 friendship.2[selection129,selection129],friendship.3[selection129,selection129]),
 dim=c(129,129,3)))

# Identify dependent behavior variables:
drinking <- sienaNet(alcohol[selection129,],type="behavior")
rock <- sienaNet(rockscale[selection129,],type="behavior")
elite <- sienaNet(elitescale[selection129,],type="behavior")
chart <- sienaNet(chartscale[selection129,],type="behavior")

# Identify constant covariate:
sexF <- coCovar(sex.F[selection129])

# Bind data together for Siena analysis:
SSWdata <- sienaDataCreate(friendship,drinking,rock,elite,chart,sexF)

# Create effects object for model specification:
SSWeffects <- getEffects(SSWdata)

# Write first descriptive results to protocol file (optional):
print01Report(SSWdata,modelname='SSW-init')

# Specify the model according to SSW:
SSWeffects <- includeEffects(SSWeffects,nbrDist2)
SSWeffects <- includeEffects(SSWeffects,egoX,altX,sameX,interaction1="sexF")
SSWeffects <- includeEffects(SSWeffects,egoX,altX,simX,interaction1="drinking")
SSWeffects <- includeEffects(SSWeffects,egoX,altX,simX,interaction1="rock")
SSWeffects <- includeEffects(SSWeffects,egoX,altX,simX,interaction1="elite")
SSWeffects <- includeEffects(SSWeffects,egoX,altX,simX,interaction1="chart")
SSWeffects <- includeEffects(SSWeffects,name="drinking",totSim,interaction1="friendship")
SSWeffects <- includeEffects(SSWeffects,name="rock",totSim,interaction1="friendship")
SSWeffects <- includeEffects(SSWeffects,name="elite",totSim,interaction1="friendship")
SSWeffects <- includeEffects(SSWeffects,name="chart",totSim,interaction1="friendship")
# Shorthand to include all effects from covariates/behaviors on behaviors:
SSWeffects$include[SSWeffects$shortName=="effFrom" & SSWeffects$type=="eval"] <- TRUE
# de-select quadratric shape parameters (included by default):
SSWeffects$include[SSWeffects$shortName=="quad"] <- FALSE
# check how model specification looks like:
SSWeffects

# Now create a model object (note that estimation option in SSW was findiff=TRUE):
SSWmodel <- sienaModelCreate(useStdInits=TRUE,projname='SSW-results')

# Estimate the model (using two cores of the processor to speed up affairs):
SSWresults <- siena07(SSWmodel,data=SSWdata,effects=SSWeffects,useCluster=TRUE,
							initC=TRUE,nbrNodes=2)

# Take a look at the results & save them:
SSWresults # Results qualitatively are the same as those reported in SSW.
save.image("SSWresults.RData")

# SSW work with an 'incomplete specification' in that they omit the 'quadratic shape'
# parameter which was found to be crucially important only after its publication.
# It therefore is important to check robustness of the SSW results upon controlling
# for this parameter!

# Add quadratic shape parameters to the model:
# Shorthand to do this for all behaviors in one command:
SSWeffects$include[SSWeffects$shortName=="quad" & SSWeffects$type=="eval"] <- TRUE
# check how model specification looks like now:
SSWeffects

# Estimate the model & take a look at the results:
SSWresults.quad.added <- siena07(SSWmodel,data=SSWdata,effects=SSWeffects,
					useCluster=TRUE, initC=TRUE,nbrNodes=2,prevAns=SSWresults)
SSWresults.quad.added
# Results suggest 'quantitative robustness' of the results reported earlier, i.e.,
# parameter values don't change very much; however, qualitatively the addition of
# the 'quad' parameter reduces significance of the peer influence effect 
# for rock and chart music (notably the former). 
# For all behaviors, the estimated social influence becomes
# somewhat smaller.
# The 'quadratic shape' effect is significant itself for
# alcohol consumption (negative sign) and elite style listening (positive sign),
# corresponding to these variables' respectively unimodal and L-shaped skewed distributions.

# Save also these results (overwriting the earlier data set):
save.image("SSWresults.RData")
